/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Summary description for BaselineSource.
	/// </summary>
	public class BaselineSource : Source
	{
		private const string Name = "Base";

		private Source source;
		private long lastPosition = 0;
		private long limit = 0;
		private int nchannel = 0;
		private int nlevel = 0;
		private int[][][] history;
		private int historyIndex;
		private int hsize;

		private int[] dataPoints = null;
		private long dataPointsPosition;

		private long lastPoint;
		private long oldMid;

		public BaselineSource(Source s)
		{
			source = s;
			if (source == null) 
			{
				throw new Exception("BaselineSource has no source");
			}
			limit = s.GetEnd();

			nchannel = s.GetNumChannels();

			nlevel = 1;
			for (Source sl = source; sl != null; nlevel++, sl = sl.GetSource());
			CheckParameters(0);
		}

		private void CheckParameters(long pos)
		{
			if (source == null) 
			{
				throw new Exception("Source closed");
			}

			hsize = 12*GetTapeParameters(pos).Gather;
			history = new int[hsize][][];
			for (int hi = 0; hi < hsize; hi++)
			{
				history[hi] = new int[nlevel][];
				for (int level = 0; level < nlevel; level++) 
				{
					history[hi][level] = new int[nchannel];
					for (int ch = 0; ch < nchannel; ch++) 
					{
						history[hi][level][ch] = 0;
					}
				}
			}
			historyIndex = 0;
			lastPoint = 0;
			oldMid = 0;
		}

		public TapeParameters GetTapeParameters() { return GetTapeParameters(GetPosition()); }
		public TapeParameters GetTapeParameters(long pos) { return source.GetTapeParameters(pos); }
		
		public string GetName() { return Name; }

		public Source GetSource() { return source; }

		public int GetNumChannels() { return source.GetNumChannels(); }

		public string GetChannelName(int channel) { return source.GetChannelName(channel); }

		public long GetStart() { return source.GetStart(); }

		public long GetEnd() { return source.GetEnd(); }

		public void SetStart(long pos) { source.SetStart(pos); }

		public void SetEnd(long pos) { source.SetEnd(pos); }

		public long GetLength() { return source.GetLength(); }

		public long GetPosition()
		{
			return source.GetPosition() - hsize;
		}

		public void SetDataPoints(int[] dataPoints, long dataPointsPosition)
		{
			this.dataPoints = dataPoints;
			this.dataPointsPosition = dataPointsPosition;
		}

		public void SetPosition(long pos)
		{
			CheckParameters(pos);
			lastPosition = pos;
			source.SetPosition(pos);
			for (int ch = 0; ch < nchannel; ch++) 
			{
				for (int hi = 0; hi < hsize; hi++) 
				{
					history[hi][0][ch] = 0;
				}
			}
			for (int i = 0; i < hsize; i++) 
			{
				Advance();
			}
		}

		public void ResetPosition()
		{
			SetPosition(lastPosition);
		}

		public long GetLimit()
		{
			return limit;
		}

		public void SetLimit(long l)
		{
			limit = l;
		}

		public void ResetLimit()
		{
			limit = source.GetEnd();
		}

		public bool AtLimit()
		{
			return GetPosition() >= limit;
		}

		public bool InBaseline(long pos)
		{
			if (dataPoints != null)
			{
				int dpi = (int)(pos - dataPointsPosition);
				if (dpi >= 0 && dpi < dataPoints.Length) 
				{
					return true;
				}
			}
			return false;
		}

		public void Advance()
		{
			for (int level = 1; level < nlevel; level++)
			{
				int[] samples = source.GetSamples(level-1);
				for (int ch = 0; ch < nchannel; ch++) 
				{
					history[historyIndex][level][ch] = samples[ch];
				}
			}

			if (dataPoints != null)
			{
				int dpi = (int)(source.GetPosition() - dataPointsPosition);
				if (dpi >= 0 && dpi < dataPoints.Length && dataPoints[dpi] != 0) 
				{
					long newPoint = source.GetPosition();
					int newData = dataPoints[dpi];
					long newMid = 0;
					int newDir = 0;
					for (int ch = 0; ch < nchannel; ch++) 
					{
						if (history[historyIndex][1][ch] > 0) 
						{
							newDir |= 1 << ch;
						}
					}
					if (lastPoint != 0) 
					{
						newMid = (newPoint + lastPoint)/2;
						if (oldMid != 0 && (int)(newPoint - oldMid) < hsize) 
						{
							DoBaseline(oldMid, newMid);
						}
					}
					lastPoint = newPoint;
					oldMid = newMid;
				}
			}
			for (int ch = 0; ch < nchannel; ch++) 
			{
				history[historyIndex][0][ch] = 0;
			}

			source.Advance();

			historyIndex++;
			if (historyIndex == hsize) 
			{
				historyIndex = 0;
			}
		}

		private void DoBaseline(long oldMid, long newMid)
		{
			long pos = source.GetPosition();
			int n;
			int pmi;
			if (oldMid == 0 || (int)(newMid - oldMid) > hsize-1) 
			{
				pmi = historyIndex+1;
				if (pmi == hsize) 
				{
					pmi = 0;
				}
				n = hsize-1;
				oldMid = 0;
			} 
			else 
			{
				pmi = historyIndex + (int)(oldMid - pos);
				if (pmi < 0) 
				{
					pmi += hsize;
				}
				n = (int)(newMid - oldMid);
			}
			int nmi = historyIndex + (int)(newMid - pos);
			if (nmi < 0)
			{
				nmi += hsize;
			}
			int m = (int)(pos - newMid);

			for (int ch = 0; ch < nchannel; ch++) 
			{
				int pms = history[pmi][0][ch];
				int nms = history[nmi][1][ch];
				int limit = GetTapeParameters().BaseLimit[ch];
				if (nms > limit) 
				{
					nms = limit;
				}
				else if (nms < -limit)
				{
					nms = -limit;
				}
				int hi = pmi;
				for (int i = 0; i <= n; i++) 
				{
					history[hi][0][ch] = (pms*(n-i) + nms*i)/n;
					hi++;
					if (hi >= hsize) 
					{
						hi = 0;
					}
				}
				for (int i = 1; i <= m; i++) 
				{
					history[hi][0][ch] = nms*(m-i)/m;
					hi++;
					if (hi >= hsize) 
					{
						hi = 0;
					}
				}
			}
		}

		public int[] GetSamples()
		{
			int[] samples = new int[nchannel];
			for (int ch = 0; ch < nchannel; ch++) 
			{
				samples[ch] = history[historyIndex][1][ch] - history[historyIndex][0][ch];
			}
			return samples;
		}

		public int[] GetSamples(int level)
		{
			if (level > 0) 
			{
				return history[historyIndex][level];
			}
			return GetSamples();
		}

		public int GetFlags()
		{
			return source.GetFlags();
		}

		public void Close()
		{
			if (source != null) 
			{
				source.Close();
			}
			source = null;
		}
	}
}
