/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Summary description for AdcSource.
	/// </summary>
	public class CrowdSource : Source
	{
		private const string Name = "Cr";

		private Source source;
		private long lastPosition = 0;
		private long limit = 0;
		private int nchannel = 0;
		private int nlevel = 0;
		private int[][][] history;
		private int historyIndex;
		private int crowd;
		private int setup;
		private int[] tcount;
		private int[] before;
		private int[] after;

		private System.IO.TextWriter log = null; //new System.IO.StreamWriter("c:\\tmp\\crowd.log");
		private int setsn = 0;

		public CrowdSource(Source s)
		{
			source = s;
			if (source == null) 
			{
				throw new Exception("CrowdSource has no source");
			}
			limit = s.GetEnd();

			nchannel = s.GetNumChannels();
			tcount = new int[nchannel];
			before = new int[nchannel];
			after = new int[nchannel];
			for (int ch = 0; ch < nchannel; ch++) 
			{
				tcount[ch] = 0;
				before[ch] = 0;
				after[ch] = 0;
			}

			nlevel = 1;
			for (Source sl = source; sl != null; nlevel++, sl = sl.GetSource());
			CheckParameters(0);
		}

		private void CheckParameters(long pos)
		{
			crowd = GetTapeParameters(pos).Crowd;
			if (crowd > 0)
			{
				int hsize = crowd+5;
				if (history == null || history.Length < hsize) 
				{
					history = new int[hsize][][];
					for (int hi = 0; hi < history.Length; hi++)
					{
						history[hi] = new int[nlevel][];
						for (int level = 0; level < nlevel; level++) 
						{
							history[hi][level] = new int[nchannel];
							for (int ch = 0; ch < nchannel; ch++) 
							{
								history[hi][level][ch] = 0;
							}
						}
					}
				}
			}
			historyIndex = 0;
		}

		public TapeParameters GetTapeParameters() { return GetTapeParameters(GetPosition()); }
		public TapeParameters GetTapeParameters(long pos) { return source.GetTapeParameters(pos); }
				
		public string GetName() { return Name; }

		public Source GetSource() { return source; }

		public int GetNumChannels() { return source.GetNumChannels(); }

		public string GetChannelName(int channel) { return source.GetChannelName(channel); }

		public long GetStart() { return source.GetStart(); }

		public long GetEnd() { return source.GetEnd(); }

		public void SetStart(long pos) { source.SetStart(pos); }

		public void SetEnd(long pos) { source.SetEnd(pos); }

		public long GetLength() { return source.GetLength(); }

		public long GetPosition()
		{
			if (crowd > 0)
			{
				return source.GetPosition() - crowd - 1;
			}
			else
			{
				return source.GetPosition();
			}
		}

		public void SetPosition(long pos)
		{
			if (log != null) 
			{
				log.WriteLine(pos + ". Set " + ++setsn);
			}
			CheckParameters(pos);
			lastPosition = pos;
			if (crowd > 0)
			{
				source.SetPosition(pos - crowd);
				for (int ch = 0; ch < nchannel; ch++) 
				{
					tcount[ch] = 0;
					before[ch] = crowd+1;
					after[ch] = crowd+1;
					for (int level = 0; level < history[0].Length; level++)
					{
						for (int hi = 0; hi < history.Length; hi++) 
						{
							history[hi][level][ch] = 0;
						}
					}
				}
				setup = crowd;
				for (; GetPosition() < pos; ) 
				{
					Advance();
				}
			}
			else
			{
				source.SetPosition(pos);
			}
		}

		public void ResetPosition()
		{
			SetPosition(lastPosition);
		}

		public long GetLimit()
		{
			return limit;
		}

		public void SetLimit(long l)
		{
			limit = l;
		}

		public void ResetLimit()
		{
			limit = source.GetEnd();
		}

		public bool AtLimit()
		{
			return GetPosition() >= limit;
		}

		public void Advance()
		{
			if (crowd > 0)
			{
				for (int level = 1; level < nlevel; level++)
				{
					int[] samples = source.GetSamples(level-1);
					for (int ch = 0; ch < nchannel; ch++) 
					{
						history[historyIndex][level][ch] = samples[ch];
					}
				}

				int pick = 0;
				for (int i = 0; i < nchannel; i++) 
				{
					int hc = historyIndex - crowd;
					if (hc < 0) 
					{
						hc += history.Length;
					}
					pick = 0;
					if (before[i] < crowd) 
					{
						pick++;
						if (before[i] < crowd/2) 
						{
							pick++;
						}
					}
					if (after[i] < crowd)
					{
						pick--;
						if (after[i] < crowd/2)
						{
							pick--;
						}
					}
					int hp = hc + pick;
					if (hp < 0) 
					{
						hp += history.Length;
					} 
					else if (hp >= history.Length) 
					{
						hp -= history.Length;
					}
					history[hc][0][i] = history[hp][1][i];

					hc = historyIndex - crowd - 3;
					if (hc < 0) 
					{
						hc += history.Length;
					}
					int hr = hc - 1;
					if (hr < 0) 
					{
						hr += history.Length;
					}
					if (setup > 0) 
					{
						if (i == nchannel-1) 
						{
							setup--;
						}
					} 
					else 
					{
						if (history[hc][1][i] != 0 && history[hc][1][i] != history[hr][1][i]) 
						{
							tcount[i]--;
							before[i] = after[i];
							if (tcount[i] > 1) 
							{
								hc += after[i];
								after[i] = 1;
								for (;;) 
								{
									hc++;
									if (hc >= history.Length) 
									{
										hc -= history.Length;
									}
									if (hc == historyIndex) 
									{
										tcount[i] = 1;
										break;
									}
									hr = hc - 1;
									if (hr < 0) 
									{
										hr += history.Length;
									}
									if (history[hc][1][i] != 0 && history[hc][1][i] != history[hr][1][i]) 
									{
										break;
									}
									after[i]++;
								}
							}
							else if (tcount[i] == 1)
							{
								after[i] = crowd + 3 - after[i];
							}
							else
							{
								after[i] = crowd + 3;
							}
						}
					}

					hc = historyIndex;
					hr = historyIndex-1;
					if (hr < 0) 
					{
						hr += history.Length;
					}
					if (history[hc][1][i] != 0 && history[hc][1][i] != history[hr][1][i]) 
					{
						tcount[i]++;
						if (tcount[i] == 1) 
						{
							after[i] = 1;
						}
					} 
					else if (tcount[i] == 1) 
					{
						after[i]++;
					}
				}

				if (log != null)
				{
					log.WriteLine(GetPosition() + "." + setsn + (setup > 0? "* " : "  ") +
						before[6] + " " + after[6] + " " + tcount[6] + " " + pick + ": " +
						GetSamples(0)[6] + " " +
						GetSamples(1)[6] + " " +
						GetSamples(2)[6] + " " +
						GetSamples(3)[6] + " " +
						GetSamples(4)[6] + "   ");
					log.Flush();
				}

				historyIndex++;
				if (historyIndex == history.Length) 
				{
					historyIndex = 0;
				}
			}
			source.Advance();
		}

		public int[] GetSamples()
		{
			return GetSamples(0);
		}

		public int[] GetSamples(int level)
		{
			if (crowd > 0)
			{
				int hc = historyIndex - crowd - 1;
				if (hc < 0) 
				{
					hc += history.Length;
				}
				return history[hc][level];
			}
			if (level == 0) 
			{
				return source.GetSamples();
			}
			return source.GetSamples(level-1);
		}

		public int GetFlags()
		{
			return source.GetFlags();
		}

		public void Close()
		{
			if (source != null) 
			{
				source.Close();
			}
			source = null;
		}
	}
}
