/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.IO;
using System.Xml;

namespace a7tlib
{
	/// <summary>
	/// Represents a collection of artifact data
	/// </summary>
	public class DataCollection
	{
		private DataFile datafile;
		private string filename;

		public DataFile DataFile { get { return datafile; } }

		public DataCollection(DataFile datafile)
		{
			this.datafile = datafile;
			string s = datafile.Filename;
			int dot = s.LastIndexOf('.');
			if (dot > 0) 
			{
				filename = s.Substring(0, dot) + ".xml";
			}
			else 
			{
				filename = s + ".xml";
			}
		}

		public DataCollection(string filename)
		{
			this.filename = filename;
			Load();
		}

		public void Write()
		{
			TextWriter ow = new StreamWriter(filename);
			ow.WriteLine("<?xml version=\"1.0\" encoding=\"utf-8\" ?>");
			ow.WriteLine("<collection xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");
			ow.WriteLine("            xsi:schemaLocation=\"http://www.pcm.org/xml/collection.xsd collection.xsd\"");
			ow.WriteLine("            xmlns=\"http://www.pcm.org/xml/collection.xsd\">");
			if (datafile.ArtifactId != null)
			{
				ow.WriteLine("  <artifact id=\"" + datafile.ArtifactId + "\"/>");
			}
			datafile.Write(ow, 2);
			ow.WriteLine("</collection>");
			ow.Close();
		}

		private void parseList(string s, int[] v)
		{
			int s0 = 0;
			int s1 = s.IndexOf(',');
			if (s1 > 0) 
			{
				for (int ch = 0; ; ch++) 
				{
					v[ch] = int.Parse(s.Substring(s0, s1 - s0));
					if (ch == datafile.Channels-1) 
					{
						break;
					}
					s0 = s1+1;
					if (ch < datafile.Channels-2) 
					{
						s1 = s.IndexOf(',', s0);
					} 
					else
					{
						s1 = s.Length;
					}
				}
			} 
			else 
			{
				int vv = int.Parse(s);
				for (int ch = 0; ch < datafile.Channels; ch++) 
				{
					v[ch] = vv;
				}
			}
		}

		private void Load()
		{
			XmlDocument doc = new XmlDocument();
			doc.Load(new FileStream(filename, FileMode.Open));

			XmlNodeList files = doc.DocumentElement.GetElementsByTagName("datafile");
			if (files.Count != 1) 
			{
				throw new Exception("Only support one file in datafile collection");
			}
			XmlElement f = (XmlElement)files[0];
			if (!"tape".Equals(f.GetAttribute("form"))) 
			{
				throw new Exception("Data file must be tape form");
			}
			datafile = new DataFile(f.GetAttribute("filename"),
				int.Parse(f.GetAttribute("channels")), -1, -1);
			bool direction = !"reverse".Equals(f.GetAttribute("direction"));

			XmlNodeList tapeparms = f.GetElementsByTagName("tapeparameters");
			int[] clip = new int[datafile.Channels];
			for (int i = 0; i < tapeparms.Count; i++) 
			{
				XmlElement tpe = (XmlElement)tapeparms[i];
				TapeParameters tp = datafile.TapeParameters.Add(long.Parse(tpe.GetAttribute("start")), long.Parse(tpe.GetAttribute("end")));
				string s = tpe.GetAttribute("skew");
				if (s != null && s.Length > 0) 
				{
					parseList(s, tp.Skew);
				}
				s = tpe.GetAttribute("width");
				if (s != null && s.Length > 0) 
				{
					tp.Width = int.Parse(s);
				}
				bool lowclip = false;
				s = tpe.GetAttribute("lowclip");
				if (s != null && s.Length > 0) 
				{
					lowclip = bool.Parse(s);
				}
				s = tpe.GetAttribute("clip");
				if (s != null && s.Length > 0) 
				{
					parseList(s, clip);
					for (int ch = 0; ch < datafile.Channels; ch++) 
					{
						tp.SetClip(clip[ch], lowclip, ch);
					}
				}
				s = tpe.GetAttribute("baselimit");
				if (s != null && s.Length > 0) 
				{
					parseList(s, tp.BaseLimit);
				} 
				else if (lowclip && !tp.Parent.LowClip) 
				{
					for (int ch = 0; ch < datafile.Channels; ch++) 
					{
						tp.BaseLimit[ch] = 3*tp.GetClip(true, ch)/2;
					}
				}
				s = tpe.GetAttribute("crowd");
				if (s != null && s.Length > 0) 
				{
					tp.Crowd = int.Parse(s);
				}
				s = tpe.GetAttribute("lowspace");
				if (s != null && s.Length > 0) 
				{
					tp.LowSpace = int.Parse(s);
				}
				s = tpe.GetAttribute("highspace");
				if (s != null && s.Length > 0) 
				{
					tp.HighSpace = int.Parse(s);
				}
				s = tpe.GetAttribute("gather");
				if (s != null && s.Length > 0) 
				{
					tp.Gather = int.Parse(s);
				}
				s = tpe.GetAttribute("maxspace");
				if (s != null && s.Length > 0) 
				{
					tp.MaxSpace = int.Parse(s);
				}
				s = tpe.GetAttribute("mingap");
				if (s != null && s.Length > 0) 
				{
					tp.MinGap = int.Parse(s);
				}
				s = tpe.GetAttribute("maxjunk");
				if (s != null && s.Length > 0) 
				{
					tp.MaxJunk = int.Parse(s);
				}
			}

			XmlNodeList records = f.GetElementsByTagName("record");
			for (int i = 0; i < records.Count; i++) 
			{
				XmlElement re = (XmlElement)records[i];
				long start = long.Parse(re.GetAttribute("start"));
				Record rec = new Record(i+1, start, long.Parse(re.GetAttribute("end")), direction);
				string s = re.GetAttribute("errors");
				if (s != null && s.Length > 0) 
				{
					rec.SetErrors(int.Parse(s));
				}
				if (datafile.TapeParameters.Count > 0) 
				{
					TapeParameters tp = datafile.TapeParameters.GetTapeParameters(start);
				}
				datafile.Records.Add(rec);
			}

			string artifactid = f.GetAttribute("artifact");
			if (artifactid != null && artifactid.Length > 0) 
			{
				datafile.ArtifactId = artifactid;
			}
			datafile.Start = long.Parse(f.GetAttribute("start"));
			datafile.End = long.Parse(f.GetAttribute("end"));
		}
	}
}
