/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.IO;

namespace a7tlib
{
	/// <summary>
	/// Represents a file of data taken from an artifact
	/// </summary>
	public class DataFile
	{
		private string artifactid;
		private string filename;
		private int nchannel;
		private long start;
		private long end;
		private RecordList records;
		private TapeParametersList tpl;

		public string ArtifactId { get { return artifactid; } set { artifactid = value; } }
		public string Filename { get { return filename; } set { filename = value; } }
		public int Channels { get { return nchannel; } set { nchannel = value; } }
		public long Start     { get { return start; }    set { start = value; init_tpl(); } }
		public long End       { get { return end; }      set { end = value; init_tpl(); } }
		public RecordList Records { get { return records; } }
		public TapeParametersList TapeParameters { get { return tpl; } }

		public DataFile(string filename, int nchannel, Int32 start, Int32 end)
		{
			this.artifactid = null;
			this.filename = filename;
			this.nchannel = nchannel;
			this.start = start;
			this.end = end;
			this.records = new RecordList();
			this.tpl = new TapeParametersList(nchannel);
			init_tpl();
		}

		internal void Write(TextWriter ow, int indent)
		{
			int direction = 0;
			for (int i = 0; i < records.Count; i++) 
			{
				if (records[i].GetMotionForward()) 
				{
					direction |= 1;
				} 
				else 
				{
					direction |= 2;
				}
			}
			string sdir = "";
			if (direction == 1) 
			{
				sdir = " direction=\"forward\"";
			}
			if (direction == 2) 
			{
				sdir = " direction=\"reverse\"";
			}

			string inds = new String(' ', indent);
			if (artifactid != null) 
			{
				ow.WriteLine("<datafile artifact=\"" + artifactid + "\"");
			}
			else 
			{
				ow.WriteLine("<datafile");
			}
			ow.WriteLine(inds + " filename=\"" + filename + "\"" +
				" form=\"tape\"" +
				" channels=\"" + nchannel + "\"" +
				sdir +
				" start=\"" + start + "\"" +
				" end=\"" + end + "\">");

			int tc = tpl.Count;
			int rc = records.Count;
			long lostart = start;
			long loend = start;
			long histart, hiend;
			for (; tc > 0 || rc > 0; ) 
			{
				Object best = null;
				histart = end;
				hiend = end;
				for (int i = 0; i < tpl.Count; i++) 
				{
					TapeParameters tp = tpl[i];
					if (tp.Start >= lostart && tp.Start <= histart &&
						(tp.Start > lostart || tp.End > loend) && (tp.Start < histart || tp.End < hiend)) 
					{
						histart = tp.Start;
						hiend = tp.End;
						best = tp;
					}
				}
				for (int i = 0; i < records.Count; i++) 
				{
					Record rec = records[i];
					if (rec.GetStart() >= lostart && rec.GetStart() <= histart &&
						(rec.GetStart() > lostart || rec.GetEnd() > loend) && (rec.GetStart() < histart || rec.GetEnd() < hiend)) 
					{
						histart = rec.GetStart();
						hiend = rec.GetEnd();
						best = rec;
					}
				}
				lostart = histart;
				loend = hiend;
				if (best == null) 
				{
					break;
				}
				if (best is TapeParameters) 
				{
					((TapeParameters)best).Write(ow, indent + 2);
					tc--;
					for (int i = 0; i < records.Count; i++) 
					{
						Record rec = records[i];
						if (rec.GetStart() == lostart && rec.GetEnd() == loend) 
						{
							best = rec;
							break;
						}
					}
				} 
				if (best is Record)
				{
					((Record)best).Write(ow, indent + 2);
					rc--;
				}
			}
			ow.WriteLine(inds + "</datafile>");
		}

		private void init_tpl()
		{
			if (tpl.Count == 0 && start >= 0 && end >= 0) 
			{
				tpl.Add(start, end);
			}
		}
	}
}
