/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Summary description for AdcSource.
	/// </summary>
	public class DirSource : Source
	{
		private const string Name = "Dir";

		private Source source;
		private long lastPosition = 0;
		private long limit = 0;
		private int nchannel = 0;
		private int nlevel = 0;
		private int[][] history;
		private int[] max;
		private int[] min;

		public DirSource(Source s)
		{
			source = s;
			if (source == null) 
			{
				throw new Exception("DirSource has no source");
			}
			if (!(source is BaselineSource)) 
			{
				throw new Exception("DirSource must be based on BaselineSource");
			}
			limit = s.GetEnd();

			nchannel = s.GetNumChannels();
			max = new int[nchannel];
			min = new int[nchannel];
			for (int ch = 0; ch < nchannel; ch++) 
			{
				max[ch] = 0;
				min[ch] = 0;
			}

			nlevel = 1;
			for (Source sl = source; sl != null; nlevel++, sl = sl.GetSource());

			history = new int[nlevel][];
			for (int level = 0; level < nlevel; level++) 
			{
				history[level] = new int[nchannel];
				for (int ch = 0; ch < nchannel; ch++) 
				{
					history[level][ch] = 0;
				}
			}
		}

		public TapeParameters GetTapeParameters() { return GetTapeParameters(GetPosition()); }
		public TapeParameters GetTapeParameters(long pos) { return source.GetTapeParameters(pos); }
		
		public string GetName() { return Name; }

		public Source GetSource() { return source; }

		public int GetNumChannels() { return source.GetNumChannels(); }

		public string GetChannelName(int channel) { return source.GetChannelName(channel); }

		public long GetStart() { return source.GetStart(); }

		public long GetEnd() { return source.GetEnd(); }

		public void SetStart(long pos) { source.SetStart(pos); }

		public void SetEnd(long pos) { source.SetEnd(pos); }

		public long GetLength() { return source.GetLength(); }

		public long GetPosition()
		{
			return source.GetPosition() - 1;
		}

		public void SetPosition(long pos)
		{
			lastPosition = pos;
			source.SetPosition(pos);
			for (int ch = 0; ch < nchannel; ch++) 
			{
				max[ch] = 0;
				min[ch] = 0;
				for (int level = 0; level < history.Length; level++)
				{
					history[level][ch] = 0;
				}
			}
			Advance();
		}

		public void ResetPosition()
		{
			SetPosition(lastPosition);
		}

		public long GetLimit()
		{
			return limit;
		}

		public void SetLimit(long l)
		{
			limit = l;
		}

		public void ResetLimit()
		{
			limit = source.GetEnd();
		}

		public bool AtLimit()
		{
			return source.GetPosition() - 1 >= limit;
		}

		public void Advance()
		{
			bool lowClip = ((BaselineSource)source).InBaseline(GetPosition());
			for (int level = 1; level < nlevel; level++)
			{
				int[] samples = source.GetSamples(level-1);
				for (int ch = 0; ch < nchannel; ch++) 
				{
					history[level][ch] = samples[ch];
				}
			}

			for (int i = 0; i < nchannel; i++) 
			{
				int dir = history[0][i];
				int samp = history[1][i];
				if (samp < 0) 
				{
					max[i] = 0;
					if (dir > 0) 
					{
						dir = 0;
					}
				} 
				else if (samp >= max[i]) 
				{
					max[i] = samp;
				}
				else if (max[i] > GetTapeParameters().GetClip(lowClip, i)) 
				{
					dir = max[i];
				} 
				if (samp > 0) 
				{
					min[i] = 0;
					if (dir < 0) 
					{
						dir = 0;
					}
				} 
				else if (samp < min[i]) 
				{
					min[i] = samp;
				} 
				else if (min[i] < -GetTapeParameters().GetClip(lowClip, i)) 
				{
					dir = min[i];
				}
				history[0][i] = dir;
			}
			source.Advance();
		}

		public int[] GetSamples()
		{
			return GetSamples(0);
		}

		public int[] GetSamples(int level)
		{
			return history[level];
		}

		public int GetFlags()
		{
			return source.GetFlags();
		}

		public void Close()
		{
			if (source != null) 
			{
				source.Close();
			}
			source = null;
		}
	}
}
