/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Summary description for LumpSource.
	/// </summary>
	public class LumpSource : Source
	{
		private const string Name = "Lump";

		private Source source;
		private long lastPosition = 0;
		private long limit = 0;
		private int nchannel;
		private int[][] history;
		private int historyIndex = 0;
		private int[] sum;

		public LumpSource(Source source)
		{
			this.source = source;
			if (source == null) 
			{
				throw new Exception("No source");
			}
			limit = source.GetEnd();

			nchannel = source.GetNumChannels();
			sum = new int[nchannel];
			for (int j = 0; j < nchannel; j++) 
			{
				sum[j] = 0;
			}
			SetPosition(0);
		}

		public TapeParameters GetTapeParameters() { return source.GetTapeParameters(); }
		public TapeParameters GetTapeParameters(long pos) { return source.GetTapeParameters(pos); }

		private void CheckParameter(long pos) 
		{
			if (pos < 0) 
			{
				pos = 0;
			}
			TapeParameters tp = source.GetTapeParameters(pos);
			if (history == null || history.Length != tp.Width)
			{
				history = new int[tp.Width][];
				for (int i = 0; i < history.Length; i++) 
				{
					history[i] = new int[nchannel];
				}
			}
		}

		public string GetName() { return Name; }

		public Source GetSource() { return source; }

		public int GetNumChannels() { return source.GetNumChannels(); }

		public string GetChannelName(int channel) { return source.GetChannelName(channel); }

		public long GetStart() { return source.GetStart(); }

		public long GetEnd() { return source.GetEnd(); }

		public void SetStart(long pos) { source.SetStart(pos); }

		public void SetEnd(long pos) { source.SetEnd(pos); }

		public long GetLength() 
		{
			return source.GetLength();
		}

		public long GetPosition()
		{
			return source.GetPosition() - history.Length/2;
		}

		public void SetPosition(long pos)
		{
			if (source == null) 
			{
				throw new Exception("Source is closed");
			}
			CheckParameter(pos);
			
			source.SetPosition(pos - history.Length/2 - 1);

			for (int i = 0; i < nchannel; i++) 
			{
				sum[i] = 0;
			}
			for (historyIndex = 0; historyIndex < history.Length; historyIndex++)
			{
				for (int ch = 0; ch < nchannel; ch++) 
				{
					history[historyIndex][ch] = 0;
				}
			}
			historyIndex = 0;
			for (; GetPosition() < pos; ) 
			{
				Advance();
			}
			lastPosition = GetPosition();
		}

		public void ResetPosition()
		{
			SetPosition(lastPosition);
		}

		public long GetLimit()
		{
			return limit;
		}

		public void SetLimit(long l)
		{
			limit = l;
		}

		public void ResetLimit()
		{
			limit = source.GetEnd();
		}

		public bool AtLimit()
		{
			return GetPosition() >= limit;
		}

		public void Advance()
		{
			historyIndex++;
			if (historyIndex == history.Length) 
			{
				historyIndex = 0;
			}
			int[] samples = source.GetSamples();
			for (int i = 0; i < nchannel; i++) 
			{
				sum[i] -= history[historyIndex][i];
				history[historyIndex][i] = samples[i];
				sum[i] += history[historyIndex][i];
			}
			source.Advance();
		}

		public int[] GetSamples()
		{
			return sum;
		}

		public int[] GetSamples(int level)
		{
			if (level > 1) 
			{
				throw new Exception("GetSamples level > 1 in LumpSource");
			}
			if (level == 0) 
			{
				return GetSamples();
			}

			int middle = historyIndex - history.Length/2;
			if (middle < 0) 
			{
				middle += history.Length;
			}
			if (history[middle] != null) 
			{
				return history[middle];
			}

			int[] samples = new int[nchannel];
			for (int i = 0; i < nchannel; i++) 
			{
				samples[i] = 0;
			}
			return samples;
		}

		public int GetFlags()
		{
			return source.GetFlags();
		}

		public void Close()
		{
			if (source != null) 
			{
				source.Close();
			}
			source = null;
		}
	}
}
