/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.IO;

namespace a7tlib
{
	/// <summary>
	/// Summary description for RawSource.
	/// </summary>
	public class RawSource : Source
	{
		public const int FlagBOT = 0x0100;
		public const int FlagEOT = 0x0200;
		public const int FlagForward = 0x0001;
		public const int FlagReverse = 0x0002;
		private const int Channels = 7;
		private const string Name = "Raw";
		private const int MaxSkew = 10;
		private const string ChannelNames = "0123456P";

		private Stream file = null;
		private DataFile datafile;
		private long startPos = 0;
		private long endPos = 0;
		private long length = 0;
		private long position = 0;
		private long lastPosition = 0;
		private long limit = 0;
		private const int sampleSize = 16;
		private byte[] buf = new byte[sampleSize];
		private int[][] history = new int[MaxSkew][];
		private int historyIndex = 0;
		private bool phase = false;

		public RawSource(DataFile datafile)
		{
			this.datafile = datafile;
			file = File.Open(datafile.Filename, FileMode.Open, FileAccess.Read, FileShare.Read);
			length = 2 * file.Length / sampleSize;
			limit = length;
			if (datafile.End > 0)
			{
				startPos = datafile.Start;
				endPos = datafile.End;
			}
			else
			{
				startPos = 0;
				endPos = length;
				datafile.Start = startPos;
				datafile.End = endPos;
			}

			for (int i = 0; i < MaxSkew; i++) 
			{
				history[i] = new int[Channels];
				for (int j = 0; j < Channels; j++) 
				{
					history[i][j] = 0;
				}
			}

			InternalSetPosition(0);
		}

		public RawSource Clone()
		{
			return new RawSource(datafile);
		}

		public string GetName() { return Name; }

		public Source GetSource() { return null; }

		public int GetNumChannels() { return Channels; }

		public string GetChannelName(int channel)
		{
			return ChannelNames.Substring(channel, 1);
		}

		public TapeParameters GetTapeParameters()
		{
			return GetTapeParameters(GetPosition());
		}

		public TapeParameters GetTapeParameters(long pos)
		{
			return datafile.TapeParameters.GetTapeParameters(pos);
		}
		
		public long GetStart() { return startPos; }

		public long GetEnd() { return endPos; }

		public void SetStart(long pos) { startPos = pos; datafile.Start = pos; }

		public void SetEnd(long pos) { endPos = pos; datafile.End = pos; }

		public long GetLength() { return length; }

		public long GetPosition()
		{
			return position;
		}

		public void SetPosition(long pos)
		{
			lastPosition = pos;
			InternalSetPosition(pos);
		}

		public void ResetPosition()
		{
			InternalSetPosition(lastPosition);
		}

		public long GetLimit()
		{
			return limit;
		}

		public void SetLimit(long l)
		{
			limit = l;
		}

		public void ResetLimit()
		{
			limit = length;
		}

		public bool AtLimit()
		{
			return position >= limit;
		}

		private void InternalSetPosition(long pos)
		{
			if (file == null) 
			{
				throw new Exception("Source is closed");
			}
			for (int i = 0; i < history.Length; i++) 
			{
				for (int j = 0; j < Channels; j++) 
				{
					history[i][j] = 0;
				}
			}

			position = pos - MaxSkew;
			long n = position/2;
			if (n < 0) 
			{
				n = 0;
			}
			file.Seek(sampleSize*n, SeekOrigin.Begin);
			file.Read(buf, 0, sampleSize);
			phase = (position & 1) != 0;

			for (historyIndex = 0; position < pos; historyIndex++) 
			{
				ReadFlat(history[historyIndex]);
			}
			historyIndex--;
		}

		private void ReadFlat(int[] samples)
		{
			if (file == null) 
			{
				throw new Exception("Source is closed");
			}
			if (position >= 0 && position >= lastPosition && position < limit)
			{
				int p = 0;
				for (int ch = 0; ch < Channels; ch++)
				{
					int v = (int)(buf[p++] & 0xFF);
					v += (int)(buf[p++]) << 8;
					v -= 0x0800;
					samples[ch] = v;
				}
				if (phase) 
				{
					file.Read(buf, 0, sampleSize);
					p = 0;
					for (int ch = 0; ch < Channels; ch++)
					{
						int v = (int)(buf[p++] & 0xFF);
						v += (int)(buf[p++]) << 8;
						v -= 0x0800;
						samples[ch] = (samples[ch] + v)/2;
					}
					phase = false;
				}
				else
				{
					phase = true;
				}
			}
			else
			{
				for (int ch = 0; ch < Channels; ch++)
				{
					samples[ch] = 0;
				}
			}
			position++;
		}

		public void Advance()
		{
			historyIndex++;
			if (historyIndex == MaxSkew) 
			{
				historyIndex = 0;
			}
			ReadFlat(history[historyIndex]);
		}

		public int[] GetSamples()
		{
			int[] samples = new int[Channels];

			for (int i = 0; i < Channels; i++) 
			{
				int skewIndex = historyIndex;
				TapeParameters tp = GetTapeParameters();
				if (tp != null)
				{
					skewIndex -= tp.Skew[i];
				}
				if (skewIndex < 0) 
				{
					skewIndex += MaxSkew;
				}
				samples[i] = history[skewIndex][i];
			}
			return samples;
		}

		public int[] GetSamples(int level)
		{
			if (level > 0) 
			{
				throw new Exception("GetSamples level > 0 in RawSource");
			}
			return GetSamples();
		}

		public int GetFlags()
		{
			int v = (int)(buf[sampleSize-2] & 0xFF);
			v += (int)(buf[sampleSize-1]) << 8;
			return ~v;
		}

		public void Close()
		{
			if (file != null) 
			{
				file.Close();
			}
			file = null;
		}
	}
}
