/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Evaluation data and results for a record.
	/// </summary>
	public class RecordInfo
	{
		private const int Scale = 1000;
		private const int MinDensity = Scale/2;

		private long position;
		private int[] transitions = null;
		private bool[] transitionError = null;
		private int[] profile = null;
		private int[] lumped = null;
		private int[] density = null;
		private int[] status = null;
		private int[] data = null;
		private int index;
		private int dataCount;
		private bool hasLRC;
		private int nchannel;
		private int datamark;
		private int datamask;
		private int[] parity;
		private int bits;
		private int transitionErrorCount;
		private int errors;
		private int fixups;
		private int goodParity = 0;
		private int start = 0;
		private int finish = 0;
		private int spacing = 0;
		private int[] autocorr = null;

		private System.IO.TextWriter log = null; // new System.IO.StreamWriter("c:\\tmp\\recordinfo.log");
		private int logsn = 0;

		public RecordInfo(long position, int length, int nchannel)
		{
			this.nchannel = nchannel;
			datamark = 1 << nchannel;
			datamask = (1 << nchannel) - 1;
			Reset(position, length);
		}

		public int GetStatus(int i)
		{
			if (i < 0 || i >= index) 
			{
				return 0;
			}
			return status[i];
		}

		public void SetTransition(int channel)
		{
			bits++;
			if (index < transitions.Length) 
			{
				transitions[index] |= 1 << channel;
			}
		}

		public void Missing()
		{
			transitionError[index] = true;
			transitionErrorCount++;
		}

		public int GetBits() { return bits; }
		public int GetMissing()
		{
			return transitionErrorCount;
		}

		public int GetCount(int i)
		{
			if (i < 0 || i >= index) 
			{
				return 0;
			}
			int count = 0;
			if (transitions[i] != 0) 
			{
				for (int j = 0; (1 << j) <= transitions[i]; j++) 
				{
					if ((transitions[i] & (1 << j)) != 0) 
					{
						count++;
					}
				}
			}
			return count;
		}

		public int GetLumped(int i)
		{
			if (i < 0 || i >= index) 
			{
				return 0;
			}
			return lumped[i];
		}

		public int GetSpacing() { return spacing; }

		public bool IsDataValid(int i)
		{
			if (i <= 0 || data == null || i >= index) 
			{
				return false;
			}
			if ((data[i] & datamark) != 0) 
			{
				return true;
			}
			return false;
		}

		public int GetData(int i)
		{
			return GetDataPoint(i) & datamask;
		}

		public int GetDataPoint(int i)
		{
			if (i <= 0 || data == null || i >= index) 
			{
				return 0;
			}
			return data[i];
		}

		public byte[] GetData(bool reverse)
		{
			if (dataCount == 0) 
			{
				return null;
			}
			int count = hasLRC? dataCount-1 : dataCount;
			byte[] db = new byte[count];

			int pos = 0;
			int incr = 1;
			int start = 0;
			int end = count-1;
			if (reverse) 
			{
				incr = -1;
				start = count-1;
				end = 0;
			}
			for (int i = start; ; i += incr) 
			{
				for (; pos < data.Length && (data[pos] == 0 || status[pos] > 4); pos++);
				if (pos >= data.Length) 
				{
					throw new Exception("RecordInfo internal error, data count too large");
				}
				db[i] = (byte)(data[pos] & datamask);
				pos++;
				if (i == end)
				{
					break;
				}
			}
			return db;
		}

		public long GetStart() { return position; }

		public int GetLength() { return index; }

		public int GetDataCount() { return dataCount; }

		public void Reset(long position, long length)
		{
			this.position = position;
			if (transitions == null || transitions.Length < length) 
			{
				transitions = new int[length];
			}
			if (transitionError == null || transitionError.Length < length)
			{
				transitionError = new bool[length];
			}
			index = 0;
			errors = 0;
			fixups = 0;
			dataCount = 0;
			hasLRC = false;
			bits = 0;
			transitionErrorCount = 0;
			transitions[0] = 0;
			transitionError[0] = false;
		}

		public void Increment()
		{
			index++;
			if (status != null && index < status.Length) 
			{
				status[index] = 0;
			}
			if (index < transitions.Length) 
			{
				transitions[index] = 0;
				transitionError[index] = false;
			}
		}

		public int GetErrors() { return errors; }

		public int GetFixups() { return fixups; }

		private void setupProfile(int gather)
		{
			if (profile != null && profile.Length == 2*gather) 
			{
				return;
			}
			profile = new int[2*gather];
			profile[0] = gather*gather;
			for (int i = 1; i < gather; i++) 
			{
				profile[i] = profile[i-1] - i;
			}
			for (int i = gather; i < 2*gather; i++) 
			{
				profile[i] = profile[i-1] - (2*gather - i);
			}
		}

		private void addProfile(int i, int scale)
		{
			scale /= profile[0];
			for (int j = 0; j < profile.Length; j++) 
			{
				if (i-j >= 0 && i-j < lumped.Length) 
				{
					lumped[i-j] += scale*profile[j];
				}
				if (j > 0 && i+j < lumped.Length) 
				{
					lumped[i+j] += scale*profile[j];
				}
			}
		}

		private void LumpTransitions(TapeParameters tp)
		{
			int basewidth = 4*tp.Gather; // even, typically a multiple of gather
			int[] thist = new int[basewidth];

			setupProfile(tp.Gather);

			for (int i = 0; i < basewidth; i++) 
			{
				thist[i] = 0;
			}
			if (lumped == null || lumped.Length < index)
			{
				lumped = new int[index];
			}
			for (int i = 0; i < index; i++) 
			{
				lumped[i] = 0;
			}
			int ith = 0;
			int baseline = 0;
			for (int i = -basewidth/2; i < index; i++) 
			{
				int count = 0;
				if (i + basewidth/2 < index && transitions[i + basewidth/2] != 0) 
				{
					for (int j = 0; (1 << j) <= transitions[i + basewidth/2]; j++) 
					{
						if ((transitions[i + basewidth/2] & (1 << j)) != 0) 
						{
							count++;
						}
					}
				}
				baseline -= thist[ith];
				baseline += count;
				thist[ith] = count;
				ith++;
				if (ith == basewidth) 
				{
					ith = 0;
				}
				if (i < 0) 
				{
					continue;
				}

				int jth = ith - basewidth/2;
				if (jth < 0) 
				{
					jth += basewidth;
				}
				count = thist[jth];
				if (count > 0) 
				{
					addProfile(i, Scale * count / baseline);
				}
			}
		}

		private void EvaluateSpacing(TapeParameters tp)
		{
			int i, j;

			if (status == null || status.Length < index)
			{
				status = new int[index];
			}
			if (data == null || data.Length < index)
			{
				data = new int[index];
			}
			for (i = 0; i < index; i++) 
			{
				data[i] = 0;
			}

			for (start = 0; start < index && transitions[start] == 0; start++);
			for (finish = index-1; finish > start && transitions[finish] == 0; finish--);
			start -= tp.Gather;
			if (start < 0) 
			{
				start = 0;
			}
			finish += tp.Gather;
			if (finish > index) 
			{
				finish = index;
			}

			if (autocorr == null || autocorr.Length < tp.MaxSpace) 
			{
				autocorr = new int[tp.MaxSpace];
			}
			for (i = 0; i < autocorr.Length; i++) 
			{
				int ac = 0;
				for (j = start; j < start + 5*tp.MaxSpace && j + i < index; j++) 
				{
					ac += lumped[j] * lumped[j+i];
				}
				autocorr[i] = ac;
				//				System.Console.Out.WriteLine(i + " " + ac);
			}
			//			System.Console.Out.WriteLine();
			for (i = 0; i < autocorr.Length-2; i++) 
			{
				autocorr[i] += 2*autocorr[i+1] + autocorr[i+2];
			}

			// Simple algorithm, look for second maximum (after the max at 0)

			for (i = tp.Gather; i < autocorr.Length-2 && autocorr[i-1] > autocorr[i]; i++);
			for (; i < autocorr.Length-2 && autocorr[i-1] <= autocorr[i]; i++);
			spacing = i;

			start = start + tp.Gather - spacing/2;
			if (start < 0) 
			{
				start = 0;
			}
			finish = finish - tp.Gather + spacing/2;
			if (finish >= index) 
			{
				finish = index-1;
			}
		}

		/* Simple hill & valley, encourage proper spacing with extra lump
		private const int Encouragement = 0;
		private void EvaluateTransitions(TapeParameters tp)
		{
			int j;
			int n;

			setupProfile(tp.Gather);

			int t = -1;
			for (j = start; j < finish; )
			{
				int jk = j;
				for (; j < finish && lumped[j] == 0; j++);
				for (; j+1 < finish && lumped[j] <= lumped[j+1]; j++);
				if (lumped[j] == 0) 
				{
					break;
				}

				data[j] = datamark;
				if (Encouragement > 0) 
				{
					if (t >= 0) 
					{
						addProfile(t, -Encouragement);
						t = -1;
					}
					addProfile(j, Encouragement);
					addProfile(j + spacing, Encouragement);
					t = j + spacing;
				}

				for (; j+1 < finish && lumped[j] > 0 && lumped[j] >= lumped[j+1]; j++);
				if (jk == j) 
				{
					break;
				}
			}
			if (t >= 0) 
			{
				addProfile(t, -Encouragement);
				t = -1;
			}
		}
		/* */

		/* Simple hill & valley, check spacing after */
		private void EvaluateTransitions(TapeParameters tp)
		{
			int j;
			int n;

			for (j = start; j < finish; )
			{
				int jk = j;
				for (; j < finish && lumped[j] == 0; j++);
				for (; j+1 < finish && lumped[j] <= lumped[j+1]; j++);
				if (lumped[j] == 0) 
				{
					break;
				}
				data[j] = datamark;
				for (; j+1 < finish && lumped[j] > 0 && lumped[j] >= lumped[j+1]; j++);
				if (jk == j) 
				{
					break;
				}
			}
		}

		/* */

		private class SimpleSortedList 
		{
			int[] list;
			int count;

			public SimpleSortedList(int size)
			{
				list = new int[size];
				count = 0;
			}

			public void Add(int v)
			{
				list[count] = v;
				for (int i = count-1; i >= 0; i--) 
				{
					if (v < list[i]) 
					{
						list[i+1] = list[i];
						list[i] = v;
					}
				}
				count++;
			}

			public int Count { get { return count; } }

			public int GetKey(int i)
			{
				return list[i];
			}
		}

		// Look for runs of 4 characters with only 2 character spacings in between
		private void EvaluateDoubles(TapeParameters tp)
		{
			int myspacing = spacing;
			int ca = -1;
			int c0 = -1;
			int c1 = -1;
			int c2 = -1;
			int c3 = -1;
			int cz;

			for (cz = start; cz < finish; ) 
			{
				if ((data[cz] & datamark) != 0) 
				{
					if (ca != -1) 
					{
						// Get the median spacing around here
						SimpleSortedList spsort = new SimpleSortedList(5);
						spsort.Add(c0 - ca);
						spsort.Add(c1 - c0);
						spsort.Add(c2 - c1);
						spsort.Add(c3 - c2);
						spsort.Add(cz - c3);
						myspacing = (int)spsort.GetKey(2);
					}
					int limit = 2*myspacing + tp.LowSpace*myspacing/100;
					if (c0 > -1 &&
						(c3 - c0) < limit &&
						(c2 - c1) < (c1 - c0) &&
						(c2 - c1) < (c3 - c2)) 
					{
						data[c1] = 0;
						data[c2] = 0;
						c2 = (c1 + c2)/2;
						data[c2] = datamark;
						fixups++;
						c1 = c0;
						c0 = ca;
						ca = -1;
						if (c0 >= start) 
						{
							continue;
						}
					}
					ca = c0;
					c0 = c1;
					c1 = c2;
					c2 = c3;
					c3 = cz;
				}
				cz++;
			}
		}
		/* */

		private bool AttemptSplit(int c0, int c1, int c2)
		{
			int c1l = c1-1;
			int c1h = c1+1;
			for (; c1l > c0 && lumped[c1l] > 0; c1l--);
			if (c1l == c0) 
			{
				return false;
			}
			for (; c1h < c2 && lumped[c1h] > 0; c1h++);
			if (c1h == c2) 
			{
				return false;
			}
			int t1 = -1;
			int t2 = -1;
			for (int i = c1l+1; i < c1h; i++) 
			{
				if (transitions[i] != 0) 
				{
					if (t1 == -1) 
					{
						t1 = i;
					} 
					else 
					{
						t2 = i;
					}
				}
			}
			if (t2 == -1) 
			{
				return false;
			}
			data[c1] = 0;
			data[t1] = datamark;
			data[t2] = datamark;
			fixups++;
			return true;
		}

		// Look for runs of 3 characters that maybe should be 4
		private void EvaluateSplits(TapeParameters tp)
		{
			int limit = spacing + tp.HighSpace*spacing / 100;
			int c0 = -1;
			int c1 = -1;
			int c2 = -1;
			int c3;

			for (c3 = start; c3 < finish; c3++) 
			{
				if ((data[c3] & datamark) != 0) 
				{
					if (c0 != -1) 
					{
						bool first = c2 - c0 > limit;
						bool second = c3 - c1 > limit;
						if (first && second) 
						{
							first = c2 - c0 > c3 - c1;
							second = c3 - c1 > c2 - c0;
						}
						if (first) 
						{
							if (AttemptSplit(c0, c1, c2)) 
							{
								c1 = -1;
							}
						}
						if (second) 
						{
							if (AttemptSplit(c1, c2, c3)) 
							{
								c1 = -1;
								c2 = -1;
							}
						}
					}
					c0 = c1;
					c1 = c2;
					c2 = c3;
				}
			}
		}

		// Look for too-large gaps between characters
		private void EvaluateGaps(TapeParameters tp)
		{
			int limit = tp.HighSpace*spacing / 100;

			int c0, c1;

			for (c0 = start; (data[c0] & datamark) == 0 && c0 < finish; c0++);
			for (c1 = c0+1; c1 < finish; c1++) 
			{
				if ((data[c1] & datamark) != 0) 
				{
					c0 = c1;
				} 
				else if (c1 - c0 > limit) 
				{
					c0 = c0 + spacing;
					data[c0] = datamark;
					if (lumped[c0] > 0) 
					{
						fixups++;
					}
				}
			}
		}

		private void EvaluateStatus(TapeParameters tp)
		{
			int i, j;
			int min;
			int mini;

			for (i = 0; i < index; i++) 
			{
				status[i] = 0;
			}
			int alternation = 0;
			min = lumped[start];
			mini = start;
			for (i = start; i < finish; i++) 
			{
				if (lumped[i] <= min) 
				{
					min = lumped[i];
					mini = i;
				}

				if ((data[i] & datamark) != 0) 
				{
					status[i] = alternation + 1;
					if (i+1 < index)
					{
						status[i+1] = alternation + 1;
					}
					if (i-1 >= mini)
					{
						status[i-1] = alternation + 1;
					}
					for (j = 1; i+j < index && lumped[i+j] > 0; j++) 
					{
						status[i+j] = alternation + 1;
					}
					for (j = 1; i-j >= mini && lumped[i-j] > 0; j++) 
					{
						status[i-j] = alternation + 1;
					}
					alternation = 2 - alternation;
					min = lumped[i+1];
					mini = i+1;
				}
			}

			// Make sure all transitions have some status
			for (i = 0; i < index; i++) 
			{
				if (status[i] == 0 && lumped[i] != 0) 
				{
					for (; status[i] == 0 && lumped[i] != 0; i++) 
					{
						status[i] = 6;
					}
					errors++;
				}
			}
		}

		public void EvaluateLRC(bool motionForward, bool lrcOK)
		{
			int count;
			int lrcpos;

			count = 0;
			int first = -1;
			int last = -1;
			int next = -1;
			lrcpos = 0;
			hasLRC = false;
			for (int i = 0; i < index; i++) 
			{
				if ((data[i] & datamark) != 0) 
				{
					count++;
					if (first == -1) 
					{
						first = i;
					}
					next = last;
					last = i;
				}
			}

			if (last != -1)
			{
				if (data[last] == datamark) 
				{
					data[last] = 0;
					count--;
					dataCount--;
					last = next;
				}
			}

			if (count > 2)
			{
				if (motionForward)
				{
					int z = 0;
					int wi = last - 1;
					if (count > 4)
					{
						int n = 0;
						for (; n < 2 && wi > 0 ; wi--) 
						{
							if ((data[wi] & datamark) != 0) 
							{
								n++;
								if (data[wi] == datamark) 
								{
									z++;
								}
							}
						}
					}
					if (z == 2) 
					{
						count--;
						lrcpos = last;
						hasLRC = true;
						for (; status[last-1] == status[last]; last--);
						for (int n = 4; n > 0; last--) 
						{
							if ((data[last] & datamark) != 0) 
							{
								if (data[last] != datamark) 
								{
									break;
								}
								data[last] = 0;
								count--;
								dataCount--;
								n--;
							}
						}
					} 
					else 
					{
						lrcpos = index-1;
					}
				} 
				else 
				{
					int z = 0;
					int wi = first + 1;
					if (count > 4)
					{
						int n = 0;
						for (; n < 2 && wi < index ; wi++) 
						{
							if ((data[wi] & datamark) != 0) 
							{
								n++;
								if (data[wi] == datamark) 
								{
									z++;
								}
							}
						}
					}
					if (z == 2) 
					{
						count--;
						lrcpos = first;
						hasLRC = true;
						for (; status[first+1] == status[first]; first++);
						for (int n = 4; n > 0; first++) 
						{
							if ((data[first] & datamark) != 0) 
							{
								if (data[first] != datamark) 
								{
									break;
								}
								data[first] = 0;
								dataCount--;
								count--;
								n--;
							}
						}
					}
					else
					{
						lrcpos = 0;
					}
				}
			}
			else if (count == 2) 
			{
				hasLRC = true;
				spacing /= 4;
				if (motionForward) 
				{
					lrcpos = last;
				} 
				else 
				{
					lrcpos = first;
				}
			}
			else 
			{
				if (first >= 0)
				{
					errors = 1;
					for (int i = first; i < status.Length && status[i] != 0; i++) 
					{
						status[i] = 6;
					}
					data[first] = 0;
				}
				dataCount = 0;
				return;
			}

			if (hasLRC) 
			{
				int a = status[lrcpos];
				for (; lrcpos > 0 && status[lrcpos] == a; lrcpos--);
				lrcpos++;
				for (; lrcpos < status.Length && status[lrcpos] == a; lrcpos++) 
				{
					if (lrcOK) 
					{
						status[lrcpos] = 5;
					} 
					else 
					{
						status[lrcpos] = 6;
					}
				}
			} 
			else if (!lrcOK)
			{
				errors++;
			}
		}

		public void EvaluateMissing()
		{
			for (int i = 0; i < index; i++) 
			{
				if (transitionError[i] && (status[i] == 1 || status[i] == 3))
				{
					int j = i;
					int refv = status[i];
					for (; j > 0 && status[j-1] == refv; j--);
					for (; j < status.Length && status[j] == refv; j++) 
					{
						status[j]++;
					}
					errors++;
				}
			}
		}

		public void SetupParity()
		{
			int np = 1 << nchannel;
			if (parity == null || parity.Length < np) 
			{
				parity = new int[np];
				for (int d = 0; d < np; d++) 
				{
					int count = 0;
					for (int b = 1; b < np; b <<= 1) 
					{
						if ((d & b) != 0) 
						{
							count++;
						}
					}
					parity[d] = count & 1;
				}
			}
		}

		public void EvaluateParity()
		{
			SetupParity();

			int sum = 0;
			int count = 0;
			for (int i = 1; i < index-1; i++) 
			{
				if ((data[i] & datamark) != 0)
				{
					if (parity[data[i] & datamask] > 0) 
					{
						sum++;
					}
					else
					{
						count++;
					}
				}
			}
			goodParity = count > sum? 0 : 1;
			for (int i = 1; i < index-1; i++) 
			{
				if ((data[i] & datamark) != 0 && (status[i] == 1 || status[i] == 3))
				{
					int d = data[i] & datamask;
					if (d == 0 || parity[d] != goodParity) 
					{
						int j = i;
						int refv = status[i];
						for (; j > 0 && status[j-1] == refv; j--);
						for (; j < status.Length && status[j] == refv; j++) 
						{
							status[j]++;
						}
						errors++;
					}
				}
			}
		}

		public void EvaluateData()
		{
			int maxi = -1;
			int accum = 0;
			for (int i = 1; i < index; i++) 
			{
				if (status[i-1] != status[i] || i == index-1)
				{
					if (maxi != -1) 
					{
						data[maxi] = accum | datamark;
						dataCount++;
					}
					maxi = -1;
					accum = transitions[i];
				}
				else
				{
					if ((accum & transitions[i]) != 0) 
					{
						if (!transitionError[i]) 
						{
							transitionError[i] = true;
							transitionErrorCount++;
						}
					}
					accum |= transitions[i];
				}
				if ((data[i] & datamark) != 0) 
				{
					maxi = i;
				}
			}
		}

		private bool AdjustBoundaries()
		{
			bool adjusted = false;
			int last = -1;
			for (int next = 0; next < index; next++) 
			{
				if ((data[next] & datamark) != 0) 
				{
					if (last >= 0 &&
						parity[data[last] & 0x7F] != goodParity &&
						parity[data[next] & 0x7F] != goodParity)
					{
						int center = (last + next) / 2;
						if (status[center] != 0) 
						{
							int incr = 1;
							int border = next;
							if (status[center] == status[next])
							{
								incr = -1;
								border = last;
								center = (last + next + 1)/2;
							}
							int edge = center;
							for (edge += incr; status[edge] == status[center]; edge += incr);
							for (edge -= incr; edge != center-incr && transitions[edge] == 0; edge -= incr);
							if (transitions[edge] != 0 &&
								(data[last] & data[next] & transitions[edge]) == 0 &&
								parity[transitions[edge]] == 1) 
							{
								data[last] ^= transitions[edge];
								data[next] ^= transitions[edge];
								for (; edge != border; edge += incr) 
								{
									status[edge] = status[border];
								}
								fixups++;
								adjusted = true;
							}
						}

					}
					last = next;
				}
			}
			if (adjusted) 
			{
				for (int i = 0; i < index; i++) 
				{
					if (status[i] == 2) 
					{
						status[i] = 1;
					}
					if (status[i] == 4) 
					{
						status[i] = 3;
					}
				}
			}
			return adjusted;
		}

		public int[] SkewOffset()
		{
			int current = -1;
			int idata = 0;
			int[] ich = new int[nchannel];
			int[] sch = new int[nchannel];
			int[] nch = new int[nchannel];
			for (int ch = 0; ch < nchannel; ch++) 
			{
				sch[ch] = 0;
				nch[ch] = 0;
			}
			for (int i = 0; i < index; i++) 
			{
				if (status[i] != 0 && status[i] != current) 
				{
					current = status[i];
					for (int ch = 0; ch < nchannel; ch++) 
					{
						ich[ch] = 0;
					}
					idata = 0;
				}
				if ((data[i] & datamark) != 0) 
				{
					idata = i;
				}
				if (transitions[i] != 0) 
				{
					for (int ch = 0; ch < nchannel; ch++)
					{
						if ((transitions[i] & (1 << ch)) != 0) 
						{
							ich[ch] = i;
						}
					}
				}
				if (current != 0 && (i+1 == index || status[i+1] != current))
				{
					if (idata != 0 && (current & 1) != 0)
					{
						for (int ch = 0; ch < nchannel; ch++) 
						{
							if (ich[ch] != 0) 
							{
								sch[ch] += ich[ch] - idata;
								nch[ch]++;
							}
						}
					}
					current = 0;
				}
			}
			for (int ch = 0; ch < nchannel; ch++) 
			{
				if (nch[ch] > 0)
				{
					if (sch[ch] >= 0) 
					{
						ich[ch] = -(sch[ch] + nch[ch]/2) / nch[ch];
					} 
					else 
					{
						ich[ch] = -(sch[ch] - nch[ch]/2) / nch[ch];
					}
				}
				else
				{
					ich[ch] = 0;
				}
			}
			return ich;
		}

		private void LogResults()
		{
			logsn++;
			int current = -1;
			int idata = 0;
			int[] ich = new int[nchannel];
			for (int i = 0; i < index; i++) 
			{
				if (status[i] != 0 && status[i] != current) 
				{
					current = status[i];
					for (int ch = 0; ch < nchannel; ch++) 
					{
						ich[ch] = 0;
					}
					idata = 0;
				}
				if ((data[i] & datamark) != 0) 
				{
					idata = i;
				}
				if (transitions[i] != 0) 
				{
					for (int ch = 0; ch < nchannel; ch++)
					{
						if ((transitions[i] & (1 << ch)) != 0) 
						{
							ich[ch] = i;
						}
					}
				}
				if (current != 0 && (i+1 == index || status[i+1] != current))
				{
					if (idata == 0) 
					{
						log.WriteLine(i + "." + logsn + ": No data");
					}
					else
					{
						string s = idata + "." + logsn;
						if ((current & 1) != 0) 
						{
							s += ": ";
						}
						else 
						{
							s += "* ";
						}
						for (int ch = 0; ch < nchannel; ch++) 
						{
							if (ich[ch] != 0) 
							{
								int t = ich[ch] - idata;
								if (t > 0) 
								{
									s += " +" + t;
								} 
								else if (t == 0) 
								{
									s += "  0";
								} 
								else 
								{
									s += " " + t;
								}
							}
							else
							{
								s += "  =";
							}
						}
						log.WriteLine(s);
					}
					current = 0;
				}
			}
			log.Flush();

			int[] sf = SkewOffset();
			log.WriteLine("skew offset " +
				sf[0] + " " +
				sf[1] + " " +
				sf[2] + " " +
				sf[3] + " " +
				sf[4] + " " +
				sf[5] + " " +
				sf[6]);
			log.Flush();
		}

		public void Evaluate(TapeParameters tp, bool motionForward, bool lrcOK)
		{
			if (index > transitions.Length) 
			{
				index = transitions.Length;
			}
			errors = 0;
			fixups = 0;
			dataCount = 0;
			LumpTransitions(tp);
			EvaluateSpacing(tp);
			if (spacing > tp.Gather)
			{
				EvaluateTransitions(tp);
				EvaluateDoubles(tp);
				// EvaluateSplits(tp); // Doesn't work well, needs fix for LRC
				EvaluateGaps(tp);

				EvaluateStatus(tp);
				EvaluateData();
				EvaluateMissing();

				int lrcErrors = errors;
				EvaluateLRC(motionForward, lrcOK);
				lrcErrors = errors - lrcErrors;

				EvaluateParity();
				if (errors > 1 && dataCount > 2)
				{
					if (AdjustBoundaries()) 
					{
						errors = lrcErrors;
						EvaluateMissing();
						EvaluateParity();
					}
				}
			}

			if (log != null) 
			{
				LogResults();
			}
		}
	}
}
