/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;

namespace a7tlib
{
	/// <summary>
	/// Summary description for SkewEval.
	/// </summary>
	public class SkewEval : Plottable
	{
		private const int MinCount = 10000;

		private class Hist 
		{
			private int nchannel;
			private int hwidth;
			private int[] bin;

			public Hist(int nchannel, int hwidth) 
			{
				this.nchannel = nchannel;
				this.hwidth = hwidth;
				bin = new int[(2*hwidth+1)*(nchannel*(nchannel-1)/2)];
				for (int i = 0; i < bin.Length; i++) 
				{
					bin[i] = 0;
				}
			}

			public int GetBin(int c1, int c2, int v)
			{
				if (v >= -hwidth && v <= hwidth)
				{
					return bin[(2*hwidth+1)*(c1+(c2*(c2-1)/2)) + v+hwidth];
				}
				return 0;
			}

			public void Hit(int c1, int c2, int v)
			{
				bin[(2*hwidth+1)*(c1+(c2*(c2-1)/2)) + v+hwidth]++;
			}
		}

		Source source;
		TapeParameters tp;
		ProgressWatcher progressWatcher;
		int progressClump;
		int nchannel;
		int hwidth;
		Hist hist;

		public SkewEval(Source s, ProgressWatcher p, int progressClump, int hwidth)
		{
			source = s;
			tp = source.GetTapeParameters();
			progressWatcher = p;
			this.progressClump = progressClump;
			this.hwidth = hwidth;
			nchannel = source.GetNumChannels();

			hist = new Hist(nchannel, hwidth);
			for (int ch = 0; ch < nchannel; ch++) 
			{
				tp.Skew[ch] = 0;
			}

			Process();
			Evaluate();
		}

		public void Process()
		{
			int[] old = new int[nchannel];
			int[] when = new int[nchannel];
			int clump = 0;

			for (int ch = 0; ch < nchannel; ch++)
			{
				old[ch] = 0;
				when[ch] = 0;
			}

			for (int now = 0; ; now++)
			{
				if (source.AtLimit())
				{
					break;
				}

				int[] samples = source.GetSamples();
				for (int ch = 0; ch < nchannel; ch++)
				{
					if (samples[ch] != 0 && samples[ch] != old[ch])
					{
						when[ch] = now;
					}
				}
				for (int ch = 0; ch < nchannel; ch++)
				{
					if (samples[ch] != 0 && samples[ch] != old[ch])
					{
						for (int c2 = 0; c2 < nchannel; c2++) 
						{
							int delta = now - when[c2];
							if (delta < hwidth) 
							{
								if (c2 < ch) 
								{
									hist.Hit(c2, ch, delta);
								}
								if (c2 > ch) 
								{
									hist.Hit(ch, c2, -delta);
								}
							}
						}
						old[ch] = samples[ch];
					}
				}

				source.Advance();
				clump++;
				if (clump == progressClump)
				{
					int min = DoneYet();
					if (progressWatcher != null)
					{
						progressWatcher.ReportProgress("skew...", min, MinCount);
					}
					clump = 0;
					if (min >= MinCount)
					{
						break;
					}
				}
			}

			if (progressWatcher != null)
			{
				progressWatcher.Finished();
				clump = 0;
			}
		}

		private int DoneYet()
		{
			int min = MinCount;
			for (int c1 = 0; c1 < nchannel; c1++)
			{
				for (int c2 = c1+1; c2 < nchannel; c2++)
				{
					int count = 0;
					for (int i = -hwidth; i <= hwidth; i++) 
					{
						count += hist.GetBin(c1, c2, i);
						if (count > MinCount) 
						{
							break;
						}
					}
					if (count < min)
					{
						min = count;
					}
				}
			}
			return min;
		}

		public void Evaluate()
		{
			double[][] dskew = new double[nchannel][];
			for (int i = 0; i < nchannel; i++) 
			{
				dskew[i] = new double[nchannel];
				for (int j = 0; j < nchannel; j++) 
				{
					dskew[i][j] = 0.0;
				}
			}

			for (int i = 0; i < nchannel-1; i++) 
			{
				for (int j = i+1; j < nchannel; j++) 
				{
					int count = 0;
					double d = 0.0;
					for (int k = -hwidth; k <= hwidth; k++) 
					{
						int v = hist.GetBin(i, j, k);
						count += v;
						d += k*v;
					}
					dskew[i][j] = d/count;
					dskew[j][i] = -dskew[i][j];
				}
			}
			double d0 = 0.0;
			for (int i = 0; i < nchannel; i++) 
			{
				double d = 0.0;
				for (int j = 0; j < nchannel; j++) 
				{
					d += dskew[i][j];
				}
				if (d < d0) 
				{
					d0 = d;
				}
			}
			for (int i = 0; i < nchannel; i++) 
			{
				double d = 0.0;
				for (int j = 0; j < nchannel; j++) 
				{
					d += dskew[i][j];
				}
				tp.Skew[i] = (int)((d - d0 + (double)nchannel/2.0)/(double)nchannel);
			}
		}

		private int[] GetTotals()
		{
			int[] totals = new int[hwidth+1];
			for (int i = 0; i <= hwidth; i++) 
			{
				int v = 0;
				for (int c1 = 0; c1 < nchannel; c1++)
				{
					for (int c2 = c1+1; c2 < nchannel; c2++)
					{
						if (i == 0)
						{
							v += hist.GetBin(c1, c2, tp.Skew[c1] - tp.Skew[c2]);
						} 
						else 
						{
							v += hist.GetBin(c1, c2,  i + tp.Skew[c1] - tp.Skew[c2]);
							v += hist.GetBin(c1, c2, -i + tp.Skew[c1] - tp.Skew[c2]);
						}
					}
				}
				totals[i] = v;
			}
			return totals;
		}

		public int GetNumDatasets() { return 1; }

		public string GetDatasetName(int which)
		{
			return "Bin";
		}

		public int[] GetDataset(int which)
		{
			if (which == 0) 
			{
				int[] domain = new int[hwidth+1];
				for (int i = 0; i <= hwidth; i++) 
				{
					domain[i] = i;
				}
				return domain;
			}
			return GetTotals();
		}

		public string GetName() { return "Skew"; }

		public string GetXLabel() { return "Level"; }

		public string GetYLabel() { return "Count"; }
		
		public bool IsXLog() { return false; }
		
		public bool IsYLog() { return false; }
		
		public bool IsXMarker() { return false; }
		
		public int GetXMarker()
		{
			return 0;
		}
	}
}
