/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.Collections;
using System.IO;

namespace a7tlib
{
	/// <summary>
	/// Summary description for TapeDecoder.
	/// </summary>
	public class TapeDecoder : MultiSource
	{
		private const int ClumpSize = 10000;
		private const int MaxRecord = 100*32768*6;
		private const int NominalRecord = 20*50000;
		private const int Pad = 50;

		private DataCollection datacollection;
		private DataFile datafile;
		private Source source;
		private int nchannel;
		private int evalErrors;
		private int evalFixups;
		private int gapJunk;
		private int[] spacingHist;
		private RecordList records { get { return datafile.Records; } }
		private ProgressWatcher progressWatcher;
		private RecordInfo ri;

		private Record currentRecord;
		private RecordInfo currentRecordInfo;

		public TapeDecoder(string filename)
		{
			int dot = filename.LastIndexOf('.');
			char[] seps = { '/', '\\' };
			if (dot >= 0 && filename.IndexOfAny(seps, dot) > dot) 
			{
				dot = -1;
			}
			if (dot < 0) 
			{
				try
				{
					OpenXml(filename + ".xml");
				} 
				catch (IOException e1) 
				{
					Console.Error.WriteLine(e1.Message);
					try
					{
						OpenData(filename + ".dat");
					}
					catch (IOException e2)
					{
						Console.Error.WriteLine(e2.Message);
						OpenData(filename);
					}
				}
			}
			else if (filename.ToLower().EndsWith(".xml")) 
			{
				OpenXml(filename);
			} 
			else 
			{
				OpenData(filename);
			}
		}

		private void OpenXml(string filename)
		{
			datacollection = new DataCollection(filename);
			datafile = datacollection.DataFile;
			initialize(new RawSource(datafile));
			for (int i = 0; i < records.Count; i++) 
			{
				evalErrors += records[i].GetErrors();
				evalFixups += records[i].GetFixups();
			}
		}

		private void OpenData(string filename)
		{
			datafile = new DataFile(filename, 7, -1, -1);
			initialize(new RawSource(datafile));
			datacollection = new DataCollection(datafile);
		}


		private TapeDecoder(Source s, DataCollection dc, DataFile datafile)
		{
			this.datafile = datafile;
			this.datacollection = dc;
			for (;;) 
			{
				if (s is RawSource) 
				{
					break;
				}
				s = s.GetSource();
				if (s == null) 
				{
					throw new Exception("TapeDecoder must be based on RawSource");
				}
			}
			initialize(((RawSource)s).Clone());
		}

		private void clearStats(bool clearJunk)
		{
			evalErrors = 0;
			evalFixups = 0;
			if (clearJunk) 
			{
				gapJunk = 0;
			}
			if (spacingHist != null) 
			{
				for (int i = 0; i < spacingHist.Length; i++) 
				{
					spacingHist[i] = 0;
				}
			}
		}

		private void initialize(RawSource rawSource)
		{
			nchannel = rawSource.GetNumChannels();
			ri = new RecordInfo(0, NominalRecord, nchannel);
			clearStats(true);

			progressWatcher = null;

			currentRecord = null;
			currentRecordInfo = null;

			LumpSource lumpSource = new LumpSource(rawSource);

			BaselineSource baseSource = new BaselineSource(lumpSource);

			DirSource dirSource = new DirSource(baseSource);

			source = new CrowdSource(dirSource);
		}

		private void copyStats(TapeDecoder t)
		{
			t.evalErrors = evalErrors;
			t.evalFixups = evalFixups;
			t.gapJunk = gapJunk;
			if (spacingHist != null) 
			{
				if (t.spacingHist == null || t.spacingHist.Length < spacingHist.Length) 
				{
					t.spacingHist = new int[spacingHist.Length];
				}
				int i;
				for (i = 0; i < spacingHist.Length; i++) 
				{
					t.spacingHist[i] = spacingHist[i];
				}
				for (; i < t.spacingHist.Length; i++) 
				{
					t.spacingHist[i] = 0;
				}
			}
		}

		public MultiSource Clone()
		{
			TapeDecoder td = new TapeDecoder(source, datacollection, datafile);
			return td;
		}

		public Source GetSource()
		{
			return source;
		}

		public DataFile GetDataFile()
		{
			return datafile;
		}

		public long GetStart() { return source.GetStart(); }

		public long GetEnd() { return source.GetEnd(); }

		public void SetStart(long pos) { source.SetStart(pos); }

		public void SetEnd(long pos) { source.SetEnd(pos); }

		public long GetLength()
		{
			return source.GetLength();
		}

		public TapeParameters GetTapeParameters()
		{
			return datafile.TapeParameters.GetTapeParameters(GetPosition());
		}

		public void Calibrate(ProgressWatcher pw)
		{
			Calibrate(pw, null);
		}

		public void Calibrate(ProgressWatcher pw, MultiSource transfer)
		{
			if (source == null) 
			{
				throw new Exception("Source is closed");
			}
			if (progressWatcher != null) 
			{
				throw new Exception("Operation already in progress");
			}
			progressWatcher = pw;

			int total = (int)source.GetEnd();

			try
			{
				if (progressWatcher != null)
				{
					progressWatcher.ReportProgress("skew", 0, total);
				}
				source.SetPosition(source.GetStart());
				source.ResetLimit();
				SkewEval skewEval = new SkewEval(source, progressWatcher, ClumpSize, source.GetTapeParameters().Width);
				if (progressWatcher != null)
				{
					progressWatcher.SetPlot(skewEval);
				}
			}
			catch (StopException)
			{
			}

			if (progressWatcher != null)
			{
				progressWatcher.Finished();
				progressWatcher = null;
			}
		}

		public void Mark(ProgressWatcher pw)
		{
			Mark(pw, null);
		}

		public void Mark(ProgressWatcher pw, MultiSource transfer)
		{
			mark(pw, transfer);
			if (pw != null) 
			{
				pw.Finished();
			}
		}

		private void mark(ProgressWatcher pw, MultiSource transfer)
		{
			if (source == null) 
			{
				throw new Exception("Source is closed");
			}
			if (progressWatcher != null) 
			{
				throw new Exception("Operation already in progress");
			}

			progressWatcher = pw;
			clearStats(true);
			records.Clear();

			for (Source s = source; s != null; s = s.GetSource())
			{
				if (s is BaselineSource)
				{
					((BaselineSource)s).SetDataPoints(null, 0);
					break;
				}
			}

			int total = (int)source.GetEnd();

			try
			{
				if (progressWatcher != null)
				{
					progressWatcher.ReportProgress("mark", 0, total);
				}
				source.SetPosition(source.GetStart());
				long gaplength = 0;
				long recStart = -1;
				long recEnd = -1;
				long lastEnd = 0;
				int clump = 0;
				int[] oldSamples = new int[nchannel];
				int hiatus = 0;
				int approxLen = 0;
				for (; !source.AtLimit(); ) 
				{
					int ch;
					int[] newSamples = source.GetSamples();
					for (ch = 0; ch < nchannel; ch++) 
					{
						oldSamples[ch] = newSamples[ch];
					}

					source.Advance();

					if (hiatus > 0) 
					{
						hiatus--;
						ch = nchannel;
					} 
					else
					{
						newSamples = source.GetSamples();
						for (ch = 0; ch < nchannel; ch++) 
						{
							if (newSamples[ch] != 0 && newSamples[ch] != oldSamples[ch])
							{
								break;
							}
						}
					}

					if (ch < nchannel)
					{
						recEnd = source.GetPosition();
						if (recStart == -1) 
						{
							recStart = recEnd;
							if (recStart > Pad) 
							{
								recStart -= Pad;
							}
							else
							{
								recStart = 0;
							}
							approxLen = 0;
						}
						approxLen++;
						hiatus = source.GetTapeParameters().Width;
						gaplength = 0;
					}
					else 
					{
						if (recStart != -1) 
						{
							TapeParameters tp = source.GetTapeParameters();
							gaplength++;
							if (gaplength == tp.MaxSpace) 
							{
								if (recStart - lastEnd > tp.MinGap || approxLen > tp.MaxJunk)
								{
									recEnd += Pad;
									if (recEnd >= source.GetEnd()) 
									{
										recEnd = source.GetEnd()-1;
									}
									CreateRecord(source, recStart, recEnd, approxLen);
									lastEnd = recEnd;
									recStart = -1;
									gaplength = 0;
								}
								else 
								{
									gapJunk++;
								}
								recStart = -1;
							}
						}
					}

					if (progressWatcher != null)
					{
						++clump;
						if (clump == ClumpSize) 
						{
							string s = records.Count + " records marked";
							progressWatcher.ReportProgress(s, (int)GetPosition(), total);
							clump = 0;
						}
					}
				}
				if (recStart != -1) 
				{
					recEnd += Pad;
					if (recEnd >= source.GetEnd()) 
					{
						recEnd = source.GetEnd()-1;
					}
					CreateRecord(source, recStart, recEnd, approxLen);
				}

				if (progressWatcher != null && clump > 0)
				{
					string s = records.Count + " records marked";
					progressWatcher.ReportProgress(s, (int)GetPosition(), total);
					clump = 0;
				}
			}
			catch (StopException)
			{
			}

			if (transfer is TapeDecoder) 
			{
				TapeDecoder t = (TapeDecoder)transfer;
				copyStats(t);
			}

			progressWatcher = null;
		}

		public void Evaluate(ProgressWatcher pw)
		{
			Evaluate(pw, null);
		}

		public void Evaluate(ProgressWatcher pw, MultiSource transfer)
		{
			if (source == null) 
			{
				throw new Exception("Source is closed");
			}
			if (progressWatcher != null) 
			{
				throw new Exception("Operation already in progress");
			}

			if (records.Count == 0) 
			{
				mark(pw, transfer);
			}
			Write(pw, null, false);

			if (transfer is TapeDecoder) 
			{
				TapeDecoder t = (TapeDecoder)transfer;
				copyStats(t);
			}
		}

		private void CreateRecord(Source source, long start, long end, int approxLen)
		{

			Source rawSource = source.GetSource();
			for (; !(rawSource is RawSource); rawSource = rawSource.GetSource());
			if (approxLen > 1 && approxLen < MaxRecord) 
			{
				Record rec = new Record(records.Count+1, start, end, (((RawSource)rawSource).GetFlags() & RawSource.FlagReverse) == 0);
				rec.SetCount(approxLen);
				records.Add(rec);
			}
		}

		public RecordInfo EvaluateRecord(Record rec)
		{
			return EvaluateRecord(rec, false);
		}

		public RecordInfo EvaluateRecord(Record rec, bool setComp)
		{
			bool knewit = true;
			bool needit = false;
			TapeParameters rtp = datafile.TapeParameters.GetTapeParameters(rec.GetStart());
			for (Source s = source; s != null; s = s.GetSource())
			{
				if (s is BaselineSource)
				{
					((BaselineSource)s).SetDataPoints(null, 0);
					break;
				}
			}
			RecordInfo ri = EvaluateRecord(rec, source);
			if (setComp && (ri.GetErrors() > 0 || ri.GetMissing() > 0)) 
			{
				knewit = rtp.LowClip;
				needit = true;
			}
			if (rtp.LowClip || needit) 
			{
				if (!knewit) 
				{
					if (rtp.Start != rec.GetStart() || rtp.End != rec.GetEnd()) 
					{
						rtp = datafile.TapeParameters.Add(rec.GetStart(), rec.GetEnd());
					}
					for (int ch = 0; ch < nchannel; ch++) 
					{
						int oldclip = rtp.GetClip(false, ch);
						rtp.SetClip(3*oldclip/8, true, ch);
						rtp.BaseLimit[ch] = oldclip/2;
					}
					int[] skewOff = ri.SkewOffset();
					int min = 100;
					for (int ch = 0; ch < nchannel; ch++) 
					{
						int t = rtp.Skew[ch] + skewOff[ch];
						if (t < min) 
						{
							min = t;
						}
					}
					for (int ch = 0; ch < nchannel; ch++) 
					{
						rtp.Skew[ch] += skewOff[ch] - min;
					}
					ri = EvaluateRecord(rec, source);
				}

				int[] dp = new int[ri.GetLength()];
				for (int si = 0; si < dp.Length; si++) 
				{
					dp[si] = ri.GetDataPoint(si);
				}
				for (Source s = source; s != null; s = s.GetSource())
				{
					if (s is BaselineSource)
					{
						((BaselineSource)s).SetDataPoints(dp, rec.GetStart());
						break;
					}
				}

				ri = EvaluateRecord(rec, source);
			}
			rec.SetCount(ri.GetDataCount());
			rec.SetErrors(ri.GetErrors());
			rec.SetFixups(ri.GetFixups());
			if (ri.GetErrors() > evalErrors) 
			{
				evalErrors = ri.GetErrors();
			}
			return ri;
		}

		private RecordInfo EvaluateRecord(Record rec, Source source)
		{
			source.SetPosition(rec.GetStart());
			source.SetLimit(rec.GetEnd());

			ri.Reset(rec.GetStart(), (int)(rec.GetEnd() - rec.GetStart()));
			int[] oldSamples = new int[nchannel];
			int[] newSamples = source.GetSamples();
			for (int i = 0; i < nchannel; i++) 
			{
				oldSamples[i] = 0;
			}
			int alternation = 0;
			for (; !source.AtLimit(); )
			{
				for (int i = 0; i < nchannel; i++) 
				{
					if (newSamples[i] != 0)
					{
						oldSamples[i] = newSamples[i];
					}
				}

				source.Advance();

				ri.Increment();

				newSamples = source.GetSamples();
				for (int i = 0; i < nchannel; i++) 
				{
					if (newSamples[i] != 0 && newSamples[i] != oldSamples[i]) 
					{
						ri.SetTransition(i);
						if (newSamples[i] > 0 && oldSamples[i] > 0 ||
							newSamples[i] < 0 && oldSamples[i] < 0) 
						{
							ri.Missing();
						}
					}
				}
			}

			bool lrcOK = true;
			for (int i = 0; i < nchannel; i++) 
			{
				if (newSamples[i] > 0) 
				{
					lrcOK = false;
					break;
				}
			}

//			System.Console.Out.WriteLine("-1 " + rec.GetNumber());
			ri.Evaluate(datafile.TapeParameters.GetTapeParameters(rec.GetStart()), rec.GetMotionForward(), lrcOK);
			return ri;
		}

		public void Write(ProgressWatcher pw)
		{
			Write(pw, null, true);
		}

		public void Write(ProgressWatcher pw, MultiSource transfer)
		{
			Write(pw, null, true);
			if (transfer is TapeDecoder) 
			{
				copyStats((TapeDecoder)transfer);
			}
		}

		public void Write(ProgressWatcher pw, TextWriter log)
		{
			Write(pw, log, true);
		}

		private void Write(ProgressWatcher pw, TextWriter log, bool actuallyWrite)
		{
			if (source == null) 
			{
				throw new Exception("Source is closed");
			}
			if (progressWatcher != null) 
			{
				throw new Exception("Operation already in progress");
			}

			bool setComp = !actuallyWrite;
			if (records.Count == 0) 
			{
				mark(pw, null);
				setComp = true;
			}

			progressWatcher = pw;

			Stream outfile = null;

			if (spacingHist == null) 
			{
				spacingHist = new int[datafile.TapeParameters[0].MaxSpace];
			}
			clearStats(false);

			try
			{
				if (progressWatcher != null)
				{
					progressWatcher.ReportProgress("", 0, records.Count);
				}
				int clump = 0;

				if (actuallyWrite)
				{
					string outfilename = datafile.Filename + ".bcd";
					int dot = datafile.Filename.LastIndexOf('.');
					if (dot > 0) 
					{
						outfilename = datafile.Filename.Substring(0, dot) + ".bcd";
					} 
					outfile = File.Open(outfilename, FileMode.Create, FileAccess.Write);
					if (outfile == null) 
					{
						throw new Exception("Can't open output file " + outfilename);
					}
				}

				int end = records.Count-1;
				int start = 0;
				int incr = 1;
				for (; start < records.Count && !((Record)records[start]).GetMotionForward(); start++) 
				{
					end = 0;
					incr = -1;
				}
				if (start > 0)
				{
					start--;
				}
				for (int rn = start; ; rn += incr) 
				{
					Record rec = (Record)records[rn];

					if (rec.GetMotionForward() && incr == -1) 
					{
						break;
					}

					RecordInfo ri = EvaluateRecord(rec, setComp);
					if (ri.GetErrors() > 0)
					{
						evalErrors += ri.GetErrors();
						if (log != null) 
						{
							log.WriteLine("Record " + rec.GetNumber() + " errors " + ri.GetErrors());
						}
					}
					evalFixups += ri.GetFixups();
					if (ri.GetSpacing() < spacingHist.Length) 
					{
						spacingHist[ri.GetSpacing()]++;
					}

					if (outfile != null)
					{
						byte[] buf = ri.GetData(incr == -1);

						if (buf != null)
						{
							if (nchannel < 8)
							{
								buf[0] |= 0x80;
							}
							outfile.Write(buf, 0, buf.Length);
						}
					}

					if (rn == end) 
					{
						break;
					}

					if (progressWatcher != null)
					{
						clump += (int)(rec.GetEnd() - rec.GetStart());
						if (clump >= ClumpSize) 
						{
							int rnf = rn - start;
							int total = end - start + 1;
							if (incr < 0) 
							{
								rnf = -rnf;
								total = -total;
							}
							string s = rnf + " records";
							progressWatcher.ReportProgress(s, rnf, total);
							clump = 0;
						}
					}
				}
			}
			catch (StopException)
			{
			}

			if (outfile != null)
			{
				outfile.Close();
			}

			if (actuallyWrite) 
			{
				datacollection.Write();
			}

			if (progressWatcher != null)
			{
				progressWatcher.Finished();
				progressWatcher = null;
			}
		}

		public void WriteInfo()
		{
			datacollection.Write();
		}

		public void Close()
		{
			if (source != null) 
			{
				source.Close();
			}
			source = null;
		}

		public void SetPosition(long position)
		{
			SetPosition(position, 0);
		}

		public void SetPosition(long position, long length)
		{
			Record rec = GetRecord(position);
			if (rec != null) 
			{
				if (currentRecord != rec) 
				{
					currentRecord = rec;
					currentRecordInfo = null;
				}
				if (currentRecordInfo == null) 
				{
					currentRecordInfo = EvaluateRecord(rec);
				}
			}
			else
			{
				currentRecord = null;
				currentRecordInfo = null;
			}

			source.SetPosition(position);
			if (length > 0)
			{
				source.SetLimit(position + length);
			} 
			else 
			{
				source.ResetLimit();
			}
		}

		public void ReEvaluate()
		{
			currentRecordInfo = null;
		}

		public long GetPosition()
		{
			return source.GetPosition();
		}

		public void Advance()
		{
			source.Advance();
		}

		public int GetErrorCount()
		{
			SetCurrent();
			if (currentRecordInfo != null) 
			{
				return currentRecordInfo.GetErrors();
			}
			return 0;
		}

		public int GetErrorTotal()
		{
			return evalErrors;
		}

		public int GetFixupTotal()
		{
			return evalFixups;
		}

		public int GetGapJunk()
		{
			return gapJunk;
		}

		public int[] GetSpacingHist()
		{
			return spacingHist;
		}

		public long GetSpacing()
		{
			SetCurrent();
			if (currentRecordInfo != null) 
			{
				return currentRecordInfo.GetSpacing();
			}
			return 0;
		}
		
		public long GetNextError()
		{
			SetCurrent();
			long pos = GetPosition();
			if (currentRecordInfo != null) 
			{
				int i = (int)(pos - currentRecord.GetStart());
				for (; i < currentRecordInfo.GetLength(); i++) 
				{
					int st = currentRecordInfo.GetStatus(i);
					if (st == 0 || (st & 1) != 0) 
					{
						break;
					}
				}
				for (; i < currentRecordInfo.GetLength(); i++) 
				{
					int st = currentRecordInfo.GetStatus(i);
					if (st != 0 && (st & 1) == 0) 
					{
						break;
					}
				}
				for (; i < currentRecordInfo.GetLength(); i++) 
				{
					int st = currentRecordInfo.GetDataPoint(i);
					if (st != 0) 
					{
						return currentRecord.GetStart() + i;
					}
				}
			}
			Record rec = currentRecord;
			for (;;) 
			{
				if (rec == null) 
				{
					rec = GetNextRecord();
				}
				else
				{
					rec = GetNextRecord(rec);
				}
				if (rec == null) 
				{
					break;
				}
				if (rec.GetErrors() == 0) 
				{
					continue;
				}
				RecordInfo ri = EvaluateRecord(rec);
				for (int i = 0; i < ri.GetLength(); i++) 
				{
					int st = ri.GetStatus(i);
					if (st != 0 && (st & 1) == 0) 
					{
						return rec.GetStart() + i;
					}
				}
			}
			return GetPosition();
		}

		public bool HasActivity() { return true; }

		private void SetCurrent()
		{
			if (currentRecord == null) 
			{
				currentRecord = GetRecord();
				currentRecordInfo = null;
				return;
			}
			long pos = GetPosition();
			if (currentRecord.GetStart() > pos || currentRecord.GetEnd() < pos) 
			{
				Record r = GetRecord(pos);
				if (r == currentRecord) 
				{
					return;
				}
				currentRecord = r;
				currentRecordInfo = null;
			}
		}

		public int GetActivityCountHere() 
		{
			SetCurrent();
			if (currentRecordInfo == null) 
			{
				return 0;
			}
			return currentRecordInfo.GetCount((int)(GetPosition() - currentRecord.GetStart()));
		}

		public int GetActivityLumpedHere() 
		{
			SetCurrent();
			if (currentRecordInfo == null) 
			{
				return 0;
			}
			return currentRecordInfo.GetLumped((int)(GetPosition() - currentRecord.GetStart()));
		}

		public int GetActivityStatusHere()
		{
			SetCurrent();
			if (currentRecordInfo == null) 
			{
				return 0;
			}
			return currentRecordInfo.GetStatus((int)(GetPosition() - currentRecord.GetStart()));
		}

		public bool HasData()
		{
			return true;
		}

		public bool IsDataValidHere() 
		{
			SetCurrent();
			if (currentRecordInfo == null) 
			{
				return false;
			}
			return currentRecordInfo.IsDataValid((int)(GetPosition() - currentRecord.GetStart()));
		}

		public int GetDataHere() 
		{
			SetCurrent();
			if (currentRecordInfo == null) 
			{
				return 0;
			}
			return currentRecordInfo.GetData((int)(GetPosition() - currentRecord.GetStart()));
		}

		public bool RecordOriented() { return true; }

		public int GetNumRecords()
		{
			if (records != null) 
			{
				return records.Count;
			}
			return 0;
		}

		public Record GetRecord(int index)
		{
			if (records.Count > 0 && index > 0 && index <= records.Count)
			{
				return (Record)records[index-1];
			}
			return null;
		}

		public Record GetRecord()
		{
			return GetRecord(GetPosition());
		}

		public Record GetRecord(long position)
		{
			Record rec = currentRecord;
			int maxspace = source.GetTapeParameters().MaxSpace;
			for (;;)
			{
				if (rec != null && rec.GetStart() - maxspace <= position && rec.GetEnd() + maxspace >= position)
				{
					return rec;
				}
				if (records.Count > 0) 
				{
					if (rec != null) 
					{
						if (position > rec.GetEnd()) 
						{
							Record r = GetRecord(rec.GetNumber()+1);
							if (r == null || position < r.GetStart() - maxspace) 
							{
								return rec;
							}
							rec = r;
							continue;
						}
						else
						{
							Record r = GetRecord(rec.GetNumber()-1);
							if (r == null || position > r.GetEnd() + maxspace) 
							{
								return rec;
							}
							rec = r;
							continue;
						}
					}
					rec = records[0];
					continue;
				}
				return null;
			}
		}

		public Record GetNextRecord()
		{
			return GetNextRecord(GetPosition());
		}

		public Record GetNextRecord(Record record)
		{
			return GetNextRecord(record.GetEnd());
		}

		public Record GetNextRecord(long position)
		{
			if (records.Count > 0) 
			{
				for (int i = 0; i < records.Count; i++) 
				{
					Record rec = records[i];
					if (rec.GetStart() > position) 
					{
						return rec;
					}
				}
			}
			
			return null;
		}

		public Record GetPreviousRecord()
		{
			return GetPreviousRecord(GetPosition());
		}

		public Record GetPreviousRecord(Record record)
		{
			return GetPreviousRecord(record.GetStart());
		}

		public Record GetPreviousRecord(long position)
		{
			if (records.Count > 0) 
			{
				for (int i = records.Count-1; i >= 0; i--) 
				{
					Record rec = records[i];
					if (rec.GetEnd() < position) 
					{
						return rec;
					}
				}
			}
			
			return null;
		}
	}
}
