/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.IO;

namespace a7tlib
{
	/// <summary>
	/// Summary description for TapeParameters.
	/// </summary>
	public class TapeParameters
	{
		private TapeParametersList tpl;
		private TapeParameters parent = null;
		private long start = 0;
		private long end = 0;
		private int nchannel = 0;
		private int[] skew = null;
		private int width = 9;
		private bool lowClip = false;
		private int defaultClip { get { return width*2048/8; } }
		private int[] clip = null;
		private int[] limit = null;
		private int lowspace = 35;
		private int highspace = 150;
		private int crowd = 13;
		private int gather = 3;
		private int maxspace = 300;
		private int mingap = 6000;
		private int maxjunk = 5;

		public TapeParameters(int nchannel, long start, long end, TapeParameters parent, TapeParametersList tpl)
		{
			this.start = start;
			this.end = end;
			this.parent = parent;
			this.nchannel = nchannel;
			this.tpl = tpl;
			skew = new int[nchannel];
			clip = new int[nchannel];
			limit = new int[nchannel];
	
			for (int i = 0; i < nchannel; i++) 
			{
				skew[i] = 0;
				clip[i] = defaultClip;
				limit[i] = clip[i];
			}
			if (parent != null) 
			{
				width = parent.width;
				lowClip = parent.lowClip;
				for (int ch = 0; ch < nchannel; ch++) 
				{
					clip[ch] = parent.clip[ch];
					skew[ch] = parent.skew[ch];
					limit[ch] = parent.limit[ch];
				}
				lowspace = parent.lowspace;
				highspace = parent.highspace;
				crowd = parent.crowd;
				gather = parent.gather;
				maxspace = parent.maxspace;
				mingap = parent.mingap;
				maxjunk = parent.maxjunk;
			}
		}

		public long Start 
		{
			get 
			{
				return start;
			}
			set
			{
				if (value <= end && (parent == null || value >= parent.start && value <= parent.end)) 
				{
					start = value;
					tpl.Invalidate();
				}
				else
				{
					throw new Exception("Invalid start position");
				}
			}
		}

		public long End
		{
			get
			{
				return end;
			}
			set
			{
				if (value >= start && (parent == null || value >= parent.start && value <= parent.end)) 
				{
					end = value;
					tpl.Invalidate();
				}
				else
				{
					throw new Exception("Invalid end position");
				}
			}
		}

		public bool LowClip 
		{
			get 
			{
				return lowClip;
			}
			set
			{
				if (CanChangeLowClip) 
				{
					lowClip = value;
				}
			}
		}

		public bool CanChangeLowClip 
		{
			get 
			{
				if (parent == null) 
				{
					return false;
				}
				return !parent.lowClip;
			}
		}

		public TapeParameters Parent { get { return parent; } set { parent = value; } }
		public int Width { get { return width; } set { width = value + ((value+1) % 2); } }
		public int LowSpace { get { return lowspace; } set { lowspace = value; } }
		public int HighSpace { get { return highspace; } set { highspace = value; } }
		public int Crowd { get { return crowd; } set { crowd = value; } }
		public int Gather { get { return gather; } set { gather = value; } }
		public int MaxSpace { get { return maxspace; } set { maxspace = value; } }
		public int MinGap { get { return mingap; } set { mingap = value; } }
		public int MaxJunk { get { return maxjunk; } set { maxjunk = value; } }
		public int[] Skew { get { return skew; } }
		public int[] BaseLimit { get { return limit; } }

		public int GetClip(bool low, int channel)
		{
			if (!lowClip || low)
			{
				return clip[channel];
			}
			for (TapeParameters tp = parent; tp != null; tp = tp.parent) 
			{
				if (!tp.lowClip) 
				{
					return tp.clip[channel];
				}
			}
			return defaultClip;
		}

		public void SetClip(int i, int channel)
		{
			clip[channel] = i;
		}

		public void SetClip(int i, bool low, int channel)
		{
			clip[channel] = i;
			lowClip = low;
		}

		internal void Write(TextWriter ow, int indent)
		{
			string inds = new String(' ', indent);
			string lasts = inds;
			ow.WriteLine(inds + "<tapeparameters start=\"" + start + "\" end=\"" + end + "\"");

			bool parentDiff = true;
			bool selfSame = true;
			for (int i = 1; i < nchannel; i++) 
			{
				if (skew[i] != skew[0]) 
				{
					selfSame = false;
				}
			}
			if (parent != null) 
			{
				parentDiff = false;
				for (int i = 0; i < nchannel; i++) 
				{
					if (skew[i] != parent.skew[i]) 
					{
						parentDiff = true;
						break;
					}
				}
			}
			if (parentDiff) 
			{
				string s = " skew=\"";
				for (int i = 0; i < nchannel; i++) 
				{
					s += skew[i];
					if (selfSame) 
					{
						break;
					}
					if (i < nchannel-1) 
					{
						s += ",";
					}
				}
				s += "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.width != width)
			{
				string s = " width=\"" + width + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			parentDiff = true;
			selfSame = true;
			for (int i = 1; i < nchannel; i++) 
			{
				if (clip[i] != clip[0]) 
				{
					selfSame = false;
				}
			}
			if (parent != null && parent.lowClip == lowClip) 
			{
				parentDiff = false;
				for (int i = 0; i < nchannel; i++) 
				{
					if (clip[i] != parent.clip[i]) 
					{
						parentDiff = true;
						break;
					}
				}
			}
			if (parentDiff) 
			{
				string s = " lowclip=\"" + lowClip + "\" clip=\"";
				for (int i = 0; i < nchannel; i++) 
				{
					s += clip[i];
					if (selfSame) 
					{
						break;
					}
					if (i < nchannel-1) 
					{
						s += ",";
					}
				}
				s += "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			parentDiff = true;
			selfSame = true;
			for (int i = 1; i < nchannel; i++) 
			{
				if (limit[i] != limit[0]) 
				{
					selfSame = false;
				}
			}
			if (parent != null) 
			{
				parentDiff = false;
				for (int i = 0; i < nchannel; i++) 
				{
					if (limit[i] != parent.limit[i]) 
					{
						parentDiff = true;
						break;
					}
				}
			}
			if (parentDiff) 
			{
				string s = " baselimit=\"";
				for (int i = 0; i < nchannel; i++) 
				{
					s += limit[i];
					if (selfSame) 
					{
						break;
					}
					if (i < nchannel-1) 
					{
						s += ",";
					}
				}
				s += "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.crowd != crowd)
			{
				string s = " crowd=\"" + crowd + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.lowspace != lowspace)
			{
				string s = " lowspace=\"" + lowspace + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.highspace != highspace)
			{
				string s = " highspace=\"" + highspace + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.gather != gather)
			{
				string s = " gather=\"" + gather + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.maxspace != maxspace)
			{
				string s = " maxspace=\"" + maxspace + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.mingap != mingap)
			{
				string s = " mingap=\"" + mingap + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			if (parent == null || parent.maxjunk != maxjunk)
			{
				string s = " maxjunk=\"" + maxjunk + "\"";
				if (lasts.Length + s.Length > 72) 
				{
					ow.WriteLine(lasts);
					lasts = inds;
				}
				lasts += s;
			}

			ow.WriteLine(lasts + " />");
		}
	}
}
