/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.Collections;

namespace a7tlib
{
	/// <summary>
	/// A list of tape parameter objects
	/// </summary>
	public class TapeParametersList
	{
		private int nchannel;
		private ArrayList tpl = new ArrayList();
		private int current1 = 0;
		private long lowEdge1 = 0;
		private long highEdge1 = -1;
		private int current2 = 0;
		private long lowEdge2 = 0;
		private long highEdge2 = -1;

		public int Count { get { return tpl.Count; } }
		public TapeParameters this [int i] { get { return (TapeParameters)tpl[i]; } }

		public TapeParametersList(int nchannel) { this.nchannel = nchannel; }

		public void Invalidate()
		{
			lowEdge1 = 0;
			highEdge1 = -1;
			lowEdge2 = 0;
			highEdge2 = -1;
		}

		public TapeParameters Add(long start, long end)
		{
			TapeParameters tpp = null;
			for (int i = 0; i < tpl.Count; i++) 
			{
				TapeParameters t = this[i];
				if (t.Start <= start && t.End >= end) 
				{
					if (tpp == null ||
						tpp.Start <= t.Start && tpp.End >= t.End) 
					{
						tpp = t;
					}
				}
				else if (t.Start <= start && t.End > start ||
				         t.Start < end && t.End >= end)
				{
					throw new Exception("Overlapping start and end of TapeParameters");
				}
			}
			TapeParameters tp = new TapeParameters(nchannel, start, end, tpp, this);
			tpl.Add(tp);
			Invalidate();
			return tp;
		}

		public void Remove(TapeParameters tpr)
		{
			TapeParameters parent = tpr.Parent;
			if (parent != null) 
			{
				for (int i = 1; i < Count; i++) 
				{
					if (this[i].Parent == tpr) 
					{
						this[i].Parent = parent;
					}
				}
				tpl.Remove(tpr);
				Invalidate();
			}
		}

		public void Clear() { tpl.Clear(); }

		public TapeParameters GetTapeParameters(long pos)
		{
			if (pos >= lowEdge1 && pos < highEdge1) 
			{
				return this[current1];
			}
			if (pos >= lowEdge2 && pos < highEdge2) 
			{
				return this[current2];
			}
			TapeParameters newTp = this[0];
			if (pos < newTp.Start || pos > newTp.End) 
			{
				return newTp;
			}
			long newLowEdge = newTp.Start;
			long newHighEdge = newTp.End;
			int newCurrent = 0;
			for (int i = 1; i < tpl.Count; i++) 
			{
				TapeParameters t = this[i];
				if (t.Start <= pos && t.End >= pos) 
				{
					if (newTp.Start <= t.Start && newTp.End >= t.End) 
					{
						newTp = t;
						newCurrent = i;
						if (newTp.Start > newLowEdge) 
						{
							newLowEdge = newTp.Start;
						}
						if (newTp.End < newHighEdge) 
						{
							newHighEdge = newTp.End;
						}
					} 
				}
				else if (t.End < pos && t.End > newLowEdge) 
				{
					newLowEdge = t.End;
				} 
				else if (t.Start > pos && t.Start < newHighEdge) 
				{
					newHighEdge = t.Start;
				}
			}
			if (highEdge1 < lowEdge1) 
			{
				lowEdge1 = newLowEdge;
				highEdge1 = newHighEdge;
				current1 = newCurrent;
			} 
			else if (highEdge2 < lowEdge2) 
			{
				lowEdge2 = newLowEdge;
				highEdge2 = newHighEdge;
				current2 = newCurrent;
			} 
			else if (newLowEdge == highEdge1) 
			{
				lowEdge2 = newLowEdge;
				highEdge2 = newHighEdge;
				current2 = newCurrent;
			} 
			else if (newLowEdge == highEdge2) 
			{
				lowEdge1 = newLowEdge;
				highEdge1 = newHighEdge;
				current1 = newCurrent;
			} 
			else 
			{
				lowEdge2 = newLowEdge;
				highEdge2 = newHighEdge;
				current2 = newCurrent;
			}
			return newTp;
		}
	}
}
