/**********************************
 *
 * Copyright (C) 2004 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 *********************************/


using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.IO;
using System.Drawing.Imaging;
using a7tlib;

namespace a7tview
{
	/// <summary>
	/// Summary description for ViewScreen.
	/// </summary>
	public class ViewScreen : Panel
	{
		public const int SideBorder = 10;
		public const int DataHeight = 16;
		public const int ActivityHeight = 16;
		public const int StatusHeight = 16;
		public const int SliderHeight = 16;

		public int ActivityZoom = 27;

		private MainView parent;
		private SourceControl[] sourceControl;
		private Colors colors;
		private string lasterr = null;

		public ViewScreen(MainView p, SourceControl[] sc, Colors colors)
		{
			this.parent = p;
			this.sourceControl = sc;
			this.colors = colors;
		}

		protected override void OnPaint(PaintEventArgs e)
		{
			try
			{
				Graphics g = e.Graphics;
				g.Clear(colors.backgroundColor);

				MultiSource multiSource = parent.GetMultiSource();
				if (multiSource != null) 
				{
					Source source = multiSource.GetSource();
				
					int zoom = parent.GetZoom();
					int expansion = 1;
					if (zoom == 1) 
					{
						expansion = 2;
					}
					else
					{
						zoom = zoom/2;
					}

					int screenWidth = this.Width - 2*SideBorder;
					int screenHeight = this.Height;
					int nchannel = 0;
					int channelHeight = 0;
					SourceControl[] boxlevel = null;
					Point[][][] loPoint = null;
					Point[][][] hiPoint = null;
					int[] dataValues = null;
					PointF[] dataPoints = null;
					Font dataFont = null;
					Brush dataBrush = null;
					int dataCount = 0;

					Pen statusPen = new Pen(colors.sliderColor);
					Pen[] statusPens = new Pen[7];
					statusPens[0] = new Pen(Color.White);
					statusPens[1] = new Pen(Color.DarkGreen);
					statusPens[2] = new Pen(Color.Red);
					statusPens[3] = new Pen(Color.LightGreen);
					statusPens[4] = new Pen(Color.Pink);
					statusPens[5] = new Pen(Color.Blue);
					statusPens[6] = new Pen(Color.Orange);

					int nlevel = 0;
					nchannel = source.GetNumChannels();

					int sliderTop = screenHeight - SliderHeight;
					int statusTop = sliderTop;
					int activityTop = statusTop;
					if (multiSource.HasActivity()) 
					{
						ActivityZoom = 1000;
						/*
						if (multiSource is TapeDecoder) 
						{
							int gather = ((TapeDecoder)multiSource).GetTapeParameters().GetGather();
						}
						*/
						statusTop = sliderTop - StatusHeight;
						activityTop = statusTop - ActivityHeight;
						g.DrawLine(statusPen, new Point(SideBorder, statusTop + StatusHeight/2-2),
							new Point(screenWidth + SideBorder, statusTop + StatusHeight/2-2));
						g.DrawLine(statusPen, new Point(SideBorder, statusTop + StatusHeight/2+2),
							new Point(screenWidth + SideBorder, statusTop + StatusHeight/2+2));
					}
					int dataTop = activityTop;
					if (multiSource.HasData()) 
					{
						dataTop = activityTop - DataHeight;
						if (zoom == 1) 
						{
							dataValues = new int[screenWidth];
							dataPoints = new PointF[screenWidth];
							dataFont = Font;
							dataBrush = new SolidBrush(Color.Black);
						}
					}

					channelHeight = (dataTop-20) / source.GetNumChannels();

					nlevel = 1;
					Source s = source.GetSource();
					for (; s != null; s = s.GetSource(), nlevel++);
					boxlevel = new SourceControl[nlevel];
					s = source;
					for (int level = 0; level < nlevel; s = s.GetSource(), level++)
					{
						boxlevel[level] = null;
						for (int j = 0; j < sourceControl.Length; j++) 
						{
							if (sourceControl[j].checkBox.Text.Equals(s.GetName())) 
							{
								boxlevel[level] = sourceControl[j];
								break;
							}
						}
					}

					loPoint = new Point[nlevel][][];
					hiPoint = new Point[nlevel][][];
					for (int level = 0; level < nlevel; level++) 
					{
						loPoint[level] = new Point[nchannel+2][];
						hiPoint[level] = new Point[nchannel+2][];
						for (int i = 0; i < nchannel+2; i++) 
						{
							loPoint[level][i] = new Point[screenWidth/expansion];
							hiPoint[level][i] = new Point[screenWidth/expansion];
							for (int j = 0; j < screenWidth/expansion; j++) 
							{
								loPoint[level][i][j] = new Point(expansion*j + SideBorder, 0);
								hiPoint[level][i][j] = new Point(expansion*j + SideBorder, 0);
							}
						}
					}

					long spos = parent.GetPosition() - parent.GetZoomWidth()/2;
					long epos = spos + parent.GetZoomWidth();

					multiSource.SetPosition(spos, epos - spos);
					int[][] newMin = new int[nlevel][];
					int[][] newMax = new int[nlevel][];
					for (int level = 0; level < nlevel; level++) 
					{
						newMin[level] = new int[nchannel+2];
						newMax[level] = new int[nchannel+2];
						for (int ch = 0; ch < nchannel+2; ch++) 
						{
							newMin[level][ch] = 0;
							newMax[level][ch] = 0;
						}
					}

					int sn = 0;
					int statusStart = 0;
					int statusPos = 0;
					for (int x = 0; x < screenWidth/expansion; x++) 
					{
						for (int level = 0; level < nlevel; level++) 
						{
							for (int ch = 0; ch < nchannel; ch++) 
							{
								newMin[level][ch] = channelHeight/2;
								newMax[level][ch] = -channelHeight/2;
							}
						}
						if (multiSource.HasActivity())
						{
							for (int ch = nchannel; ch < nchannel+2; ch++) 
							{
								newMin[0][ch] = ActivityHeight;
								newMax[0][ch] = 0;
							}
						}
						for (int i = 0; i < zoom; i++) 
						{
							for (int level = 0; level < nlevel; level++) 
							{
								int[] samples = source.GetSamples(level);
								for (int ch = 0; ch < nchannel; ch++) 
								{
									int v = samples[ch]/(2*boxlevel[level].zoom / channelHeight);

									if (v < newMin[level][ch]) 
									{
										if (v > -channelHeight/2) 
										{
											newMin[level][ch] = v;
										}
										else
										{
											newMin[level][ch] = -channelHeight/2;
										}
									}
									if (v > newMax[level][ch]) 
									{
										if (v < channelHeight/2) 
										{
											newMax[level][ch] = v;
										}
										else
										{
											newMax[level][ch] = channelHeight/2;
										}
									}
								}
							}

							if (multiSource.HasActivity())
							{
								for (int ch = nchannel; ch < nchannel+2; ch++) 
								{
									int v;
									if (ch == nchannel) 
									{
										v = ActivityHeight*multiSource.GetActivityCountHere()/(nchannel);
									}
									else
									{
										v = ActivityHeight*multiSource.GetActivityLumpedHere()/ActivityZoom;
									}

									if (v < newMin[0][ch]) 
									{
										if (v > -channelHeight/2) 
										{
											newMin[0][ch] = v;
										}
										else
										{
											newMin[0][ch] = -channelHeight/2;
										}
									}
									if (v > newMax[0][ch]) 
									{
										if (v < channelHeight/2) 
										{
											newMax[0][ch] = v;
										}
										else
										{
											newMax[0][ch] = channelHeight/2;
										}
									}
								}

								int status = multiSource.GetActivityStatusHere();
								if (status != sn) 
								{
									if (sn != 0) 
									{
										Pen pen = statusPens[sn];
										int sX = SideBorder + statusStart*expansion/zoom;
										int eX = SideBorder + (statusPos-1)*expansion/zoom;
										g.DrawLine(pen, new Point(sX, statusTop + StatusHeight/2-1),
											new Point(eX, statusTop + StatusHeight/2-1));
										g.DrawLine(pen, new Point(sX, statusTop + StatusHeight/2),
											new Point(eX, statusTop + StatusHeight/2));
										g.DrawLine(pen, new Point(sX, statusTop + StatusHeight/2+1),
											new Point(eX, statusTop + StatusHeight/2+1));
									}
									statusStart = statusPos;
									sn = status;
								}
								statusPos++;
							}

							if (dataValues != null && multiSource.IsDataValidHere())
							{
								dataValues[dataCount] = multiSource.GetDataHere();
								dataPoints[dataCount] =
									new PointF(loPoint[0][0][x].X - dataFont.Size, dataTop);
								dataCount++;
							}

							multiSource.Advance();
						}

						for (int level = 0; level < nlevel; level++) 
						{
							for (int ch = 0; ch < nchannel; ch++) 
							{
								loPoint[level][ch][x].Y = 10 + channelHeight*(6-ch) + channelHeight/2 - newMin[level][ch];
								hiPoint[level][ch][x].Y = 10 + channelHeight*(6-ch) + channelHeight/2 - newMax[level][ch];
							}
						}
						if (multiSource.HasActivity())
						{
							for (int ch = nchannel; ch < nchannel+2; ch++) 
							{
								loPoint[0][ch][x].Y = activityTop + ActivityHeight - newMin[0][ch];
								hiPoint[0][ch][x].Y = activityTop + ActivityHeight - newMax[0][ch];
							}
						}
					}

					multiSource.SetPosition(parent.GetPosition());
					if (parent.CenterEnabled()) 
					{
						Pen centerPen = new Pen(colors.centerColor);
						g.DrawLine(centerPen, new Point(SideBorder + screenWidth/2, 0), new Point(SideBorder + screenWidth/2, screenHeight-1));
					}
					if (parent.ZeroEnabled()) 
					{
						Pen zeroPen = new Pen(colors.zeroColor);
						for (int ch = 0; ch < nchannel; ch++) 
						{
							g.DrawLine(zeroPen,
								new Point(SideBorder, 10 + channelHeight*(6-ch) + channelHeight/2),
								new Point(SideBorder + screenWidth, 10 + channelHeight*(6-ch) + channelHeight/2));
						}
						g.DrawLine(zeroPen,
							new Point(SideBorder, activityTop + ActivityHeight),
							new Point(SideBorder + screenWidth, activityTop + ActivityHeight));
					}
					if (parent.ClipEnabled() && multiSource is TapeDecoder) 
					{
						Pen clipPen = new Pen(colors.clipColor);
						for (int ch = 0; ch < nchannel; ch++) 
						{
							int v = ((TapeDecoder)multiSource).GetTapeParameters().GetClip(true, ch) / (2*sourceControl[1].zoom / channelHeight);
							if (v < channelHeight/2) 
							{
								g.DrawLine(clipPen,
									new Point(SideBorder, 10 + channelHeight*(6-ch) + channelHeight/2 + v),
									new Point(SideBorder + screenWidth, 10 + channelHeight*(6-ch) + channelHeight/2 + v));
								g.DrawLine(clipPen,
									new Point(SideBorder, 10 + channelHeight*(6-ch) + channelHeight/2 - v),
									new Point(SideBorder + screenWidth, 10 + channelHeight*(6-ch) + channelHeight/2 - v));
							}
						}
					}

					if (multiSource.HasActivity())
					{
						Pen activityCountPen = null;
						Pen activityLumpedPen = null;
						for (int level = nlevel-1; level >= 0; level--)
						{
							Pen dataPen = new Pen(boxlevel[level].color);
							if (level == nlevel-1)
							{
								activityCountPen = dataPen;
							}
							if (level == nlevel-2)
							{
								activityLumpedPen = dataPen;
							}
							for (int ch = 0; ch < nchannel; ch++) 
							{
								if (boxlevel[level].checkBox.Checked) 
								{
									g.DrawLines(dataPen, loPoint[level][ch]);
									g.DrawLines(dataPen, hiPoint[level][ch]);
								}
							}
						}
						g.DrawLines(activityCountPen, loPoint[0][nchannel]);
						g.DrawLines(activityLumpedPen, hiPoint[0][nchannel+1]);
					}

					if (dataValues != null) 
					{
						for (int i = 0; i < dataCount; i++) 
						{
							g.DrawString(dataValues[i].ToString("X2"), dataFont, dataBrush, dataPoints[i]);
						}
					}

					Pen sliderPen = new Pen(colors.sliderColor);
					Pen validPen = new Pen(colors.centerColor);
					int sliderX = SideBorder + (int)(screenWidth*(parent.GetPosition())/source.GetLength());
					int sposX = SideBorder + (int)(screenWidth*spos/source.GetLength());
					int eposX = SideBorder + (int)(screenWidth*epos/source.GetLength());
					int svposX = SideBorder + (int)(screenWidth*source.GetStart()/source.GetLength());
					int evposX = SideBorder + (int)(screenWidth*source.GetEnd()/source.GetLength());
					g.DrawLine(sliderPen, new Point(SideBorder, sliderTop + SliderHeight/2-1),
						new Point(screenWidth + SideBorder, sliderTop + SliderHeight/2-1));
					g.DrawLine(sliderPen, new Point(SideBorder, sliderTop + SliderHeight/2+1),
						new Point(screenWidth + SideBorder, sliderTop + SliderHeight/2+1));
					g.DrawLine(validPen, new Point(svposX, sliderTop + SliderHeight/2),
						new Point(evposX, sliderTop + SliderHeight/2));
					g.DrawLine(sliderPen, new Point(sposX, sliderTop + SliderHeight/2),
						new Point(eposX, sliderTop + SliderHeight/2));
					g.DrawLine(sliderPen, new Point(sliderX, sliderTop + 2),
						new Point(sliderX, sliderTop + SliderHeight - 2));
				}

				g.Flush();
			}
			catch (Exception ex)
			{
				lasterr = ex.ToString() + ex.StackTrace;
			}
		}
	}
}
