/**********************************
 *
 * Modifications Copyright (C) 2003 Paul Pierce
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 **********************************
 *
 * This is a data acquisition program for use with a Data Translation DT-3010
 * board to obtain data from a Pertec-interface 7-track 1/2 inch magnetic
 * tape drive running at 25 ips. It is a minimal modification of the CEXAMPLE
 * program that comes with the board. Many of the original program's features
 * remain but are disabled or not needed. For instance, all the original
 * configuration features remain but are not needed because there is a builtin
 * default configuration that is correct for the intended application.
 *
 * In normal use the program is nearly completely automatic. To begin taking
 * data, click the Start button. If the tape is at load point it will run forward
 * to the end of data (several feet of blank tape, or a sufficiently long
 * tape test pattern of ones in all channels, or a few feet past EOT.)
 * Otherwise it will run in reverse to BOT. Data is written to a file
 * named "adcNNNNN.dat" where NNNNN is the lowest integer (with leading zeroes)
 * for which the file does not already exist.
 *
 * If something goes wrong it stops the tape and presents a dialog box with
 * an error message. In any case when data acquisition is done the tape stops
 * and it presents a dialog box showing the file name.
 */

/*-----------------------------------------------------------------------

 Copyright  (C)  1993-2000.   Data  Translation,  Inc.,  100  Locke   Drive,
 Marlboro Massachusetts 01752-1192.

 All rights reserved.  This software is furnished to purchaser under  a
 license  for  use  on a single computer system and can be copied (with
 the inclusion of DTI's copyright notice) only for use in such  system,
 except  as  may  be otherwise provided in writing by Data Translation,
 Inc., 100 Locke Drive, Marlboro, Massachusetts 01752-1192.

 The information in this document is subject to change  without  notice
 and  should not be construed as a commitment by Data Translation, Inc.
 Data Translation, Inc.  assumes no responsibility for any errors  that
 may appear in this document.

 Data Translation cannot assume any responsibility for the use  of  any
 portion  of  this  software  on  any  equipment  not  supplied by Data
 Translation, Inc.

PROGRAM: cexmpl.c

PURPOSE:
    Open Layers data acquisition example, shows:
        continuous ADC & DAC operation,
        single value ADC & DAC,
        Counter Timer operation.
    Continuous DAC operation uses a single buffer to generate a repetitive
    waveform.  The ADC uses multiple buffers in either burst or "run forever"
    mode.  DAC and ADC operation can be run simultaneously (if supported
    by driver.)  In addition, the program can optionally plot and/or write
    input data to a file.  The output waveform can also be viewed.

FUNCTIONS:

startup/main functions:
    WinMain()           calls initialization function, processes message loop
    InitApplication()   initializes window data and registers window
    InitInstance()      saves instance handle and creates main window

    MainWndProc()       cexmpl window function

About dialog box
    About()             "About" dialog box function

Open Board dialog box
    BoardBox()          "Select Board" dialog box function
    AddBoardList()      callback to fill board name combo box

Support functions for startup & board open 
    InitConfig()        open/init configuration for default board
    UpdateConfig()      open/init configuration for specified board
    InitBoard()         callback to find default board
    CloseBoard()        close current board

Input & Output dialog boxes
    InputBox()          "Input Options" dialog box function
    OutputBox()         "Output Options" dialog box function
    InitIOBox()         init dialog based on current hardware configuration
    SaveIOBox()         set hardware configuration based on dialog settings
    AddRangeList()      callback to fill range combo box

Acquisition dialog box
    AcqBox()            "Acquisition Options" dialog box function
    LoadAcqBox()        init dialog box based on config structure
    SaveAcqBox()        save dialog box data to config structure
    SetWaveFreq()       calculate exact output freq & buffers size
                        for continuous output phase

Control / support for continuous operation
    GenerateSignal()    generate output waveform
    PaintGraph()        update window display
    StartAcq()          start continuous operation
    StopAcq()           stop continuous operation
    WriteBufferToFile() write input data to file

Channel gain list dialog box
    CglBox()            "Input Channel/Gain List" dialog box function
    GetCGLString()      format string for CGL list box

Single value dialog box
    SvBox()             "Single Value" dialog box function

Support for CGL & single value dialog boxes
    InitChannelComboBox() callback to load channel combo box
    AddGainList()       callback to load gain combo box

Counter timer dialog box
    CTBox()             "Counter / Timer" dialog box function
    SetCTMode()         Sets CT mode & Grays Dialog box options

****************************************************************************/

#include <windows.h>     
#include "cexmpl.h"
#include <graphs.h>
#include <stdio.h>
#include <stdlib.h>       
#include <string.h>
#include <math.h>

#define FREQ_FORMAT "%.1lf"     // format used of printing frequency numbers
#define RANGE_FORMAT "%.1lfV to %.1lfV" // format used for printing input/output ranges
#define GAIN_FORMAT "%.1lf"     // format used for printing gain numbers
#define FILTER_FORMAT "%.1lfHz"     // format used for printing gain numbers
#define WIDTH_FORMAT "%.1lf"     // format used for printing pulse width numbers
#define MAXCGLSIZE 600          // max CGL size this app can handle


// Tape drive command/status

#define DRIVE_BOT	0x01
#define DRIVE_EOT	0x02
#define DRIVE_REWINDING 0x04
#define DRIVE_ONLINE	0x08
#define DRIVE_READY	0x10

#define DRIVE_FWD	0xFE
#define DRIVE_REV	0xFD
#define DRIVE_RWD	0xFB
#define DRIVE_OFFL	0xF7

unsigned long normalTimeout = 250000;
unsigned long eotTimeout = 125000;
unsigned long currentTimeout = -1;
unsigned long activityTimer;
unsigned long currentPolarity;
int startup;
int offset[7];

BOOL getDriveReady();
BOOL getDriveBot();
VOID setDriveForward();
VOID setDriveReverse();
VOID setDriveStop();

BOOL forward;
UINT driveState;
INT stopCounter;


// global data
HANDLE hInst;               /* current instance              */
CONFIG config;              // configuration data 

UINT adelement=0;

#define STRLEN 80           // string size for general text manipulation
char str[STRLEN];           // global string for general text manipulation
char adcfile[STRLEN];		// output file name for data capture
BOOL adcfilestart=TRUE;		// starting new acquisition, need new file name

#define SHOW_ERROR(hwnd,status) MessageBox(hwnd,olDaGetErrorString(status,str,STRLEN),\
                                "Error", MB_ICONEXCLAMATION | MB_OK);

// internal function prototypes

int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int);

LRESULT CALLBACK MainWndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );

BOOL CALLBACK AboutBox(  HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK BoardBox(  HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK InputBox(  HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK OutputBox( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK AcqBox(    HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK SvBox(     HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK CglBox(    HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK CTBox(     HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK DigIOBox(  HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );

BOOL CALLBACK AddBoardList( LPSTR lpszName, LPSTR lpszEntry, LPARAM lParam );
BOOL CALLBACK AddRangeList( UINT cap, DBL max, DBL min, LPARAM lParam );
BOOL CALLBACK AddGainList( UINT cap, DBL gain, DBL unused, LPARAM lParam );
BOOL CALLBACK AddFilterList( UINT cap, DBL filter, DBL unused, LPARAM lParam );
BOOL CALLBACK InitBoard( LPSTR lpszName,  LPSTR lpszEntry,  LPARAM lpconfig );

BOOL InitApplication( HANDLE );
BOOL InitInstance( HANDLE, int );
BOOL InitConfig( LPCONFIG lpconfig );
VOID InitDriveChannelList();
BOOL WriteBufferToFile( HWND hWnd, HBUF hBuf, LPSTR lpstr, UINT strlen );
BOOL AdcfileUpdate();

VOID SetAvailableOptions(HWND);
VOID CloseBoard(LPCONFIG lpconfig);
VOID SaveIOBox(HWND hDlg, HDASS hDass, LPCONFIG lpconfig);
VOID LoadAcqBox(HWND hDlg,LPCONFIG lpconfig);
VOID SaveAcqBox(HWND hDlg,LPCONFIG lpconfig);
VOID PaintGraph (HWND hwnd,LPCONFIG lpconfig);
VOID StopAcq(LPCONFIG lpconfig);
VOID GetCGLString(LPCONFIG lpconfig, UINT entry, NPSTR npstr);
VOID InitChannelComboBox(HWND hWndCombo, HDASS hDass);
VOID SetCTMode(HWND hDlg, UINT index);

ECODE UpdateConfig(LPCONFIG lpconfig,LPSTR lpszBoardName);
ECODE InitIOBox(HWND hDlg, HDASS hDass, LPCONFIG lpconfig, BOOL fShowScan);
ECODE GenerateSignal (LPCONFIG lpconfig);
ECODE StartAcq(LPCONFIG lpconfig);

DBL SetWaveFreq(DBL dDesiredFreq, DBL dSampFreq, LPLNG lplBufSize);

#define CHECKERROR( ecode )                         \
do                                                  \
{                                                   \
   ECODE olStatus;                                  \
   if( OLSUCCESS != ( olStatus = ( ecode ) ) )      \
      return olStatus;                              \
}                                                   \
while(0)      

int WINAPI WinMain( HINSTANCE hInstance,             // current instance         
                    HINSTANCE hPrevInstance,         // previous instance        
                    LPSTR lpCmdLine,                 // command line             
                    int nCmdShow )                   // show-window type (open/icon) 
/*++
This is our apps entry point.  It calls the initialization functions,
and executes the message loop.
--*/
{
    MSG msg;   // message                                

    // Increase the our message queue size so
    // we don't lose any data acq messages
    //
    SetMessageQueue(50);        

    // Get element number
    if (lpCmdLine[0]!=0)
     adelement = atoi(lpCmdLine);
    else
     adelement = 0;


    // Initialize shared things
    if( !InitApplication( hInstance ) )    
        return FALSE;     

    // Perform initializations that apply to a specific instance 
    //
    if( !InitInstance( hInstance, nCmdShow ) )
        return FALSE;

    // Acquire and dispatch messages until a WM_QUIT message is received. 


    while( GetMessage( &msg,     // message structure              
                       NULL,     // handle of window receiving the message 
                       0,        // lowest message to examine          
                       0 ) )     // highest message to examine         
    {
       TranslateMessage(&msg);   // Translates virtual key codes       
       DispatchMessage(&msg);    // Dispatches message to window       
    }
    return (msg.wParam);         // Returns the value from PostQuitMessage 
}


BOOL 
InitApplication( HANDLE hInstance )
/*++
This function initializes window data and registers the window class
--*/
{
    WNDCLASS  wc;

    // Fill in window class structure with parameters that describe the       
    // main window.                                                           

    wc.style = 0;                                      // Class style(s).                    
    wc.lpfnWndProc = MainWndProc;                      // Function to retrieve messages for  
                                                       // windows of this class.             
    wc.cbClsExtra = 0;                                 // No per-class extra data.           
    wc.cbWndExtra = 0;                                 // No window extra data 
    wc.hInstance = hInstance;                          // Application that owns the class.   
    wc.hIcon = LoadIcon(hInstance, "OpenLayersIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    
    wc.lpszMenuName =  "CexmplMenu";                   // Name of menu resource in .RC file. 
    wc.lpszClassName = "CexmplWClass";                 // Name used in call to CreateWindow. 

	lstrcpy(adcfile, "adc00000.dat");
	driveState = 0;

    // Register the window class and return success/failure code. 
    //
    return (RegisterClass(&wc));
}


BOOL 
InitInstance( HANDLE hInstance, int nCmdShow )
/*++
This function is called at initialization time for every instance of 
this application.  This function performs initialization tasks that 
cannot be shared by multiple instances.  It saves the instance handle and
creates the main window  .
--*/
{
    HWND  hWnd;                            

    // Save the instance handle in static variable, which will be used in  
    // many subsequence calls from this application to Windows.            
    //
    hInst = hInstance;

    // Create a main window for this application instance.  
    //
    hWnd = CreateWindow(
        "CexmplWClass",                 // See RegisterClass() call.          
        "7-Track Tape Data",			// Text for window title bar.         
        WS_OVERLAPPEDWINDOW,            // Window style.                      
        CW_USEDEFAULT,                  // Default horizontal position.       
        CW_USEDEFAULT,                  // Default vertical position.         
        CW_USEDEFAULT,                  // Default width.                     
        CW_USEDEFAULT,                  // Default height.                    
        NULL,                           // Overlapped windows have no parent. 
        NULL,                           // Use the window class menu.         
        hInstance,                      // This instance owns this window.    
        NULL                            // Pointer not needed.                
    );

    // If window could not be created, return "failure" 

    if( !hWnd )
       return FALSE;

    // Make the window visible; update its client area; and return "success" 
    //
    ShowWindow(hWnd, nCmdShow);  // Show the window                        
    UpdateWindow(hWnd);          // Sends WM_PAINT message                 
    return TRUE;               // Returns the value from PostQuitMessage 

}


VOID 
SetAvailableOptions( HWND hWnd )
/* 
This function enables or grays menu items depending on
the existance of certain subsystem types.
*/
{
    UINT uEnable;
    UINT uicap=0;
    ECODE status;

    HMENU hMenu = GetMenu( hWnd );             // get handle to main menu
    HMENU hSubMenu = GetSubMenu( hMenu, 0 );   // get handle to sub menu

    // if NO A/D or D/A subsystem...
    //    
    if( ( !config.hAdc ) && ( !config.hDac ) ) 
       uEnable = MF_BYPOSITION | MF_GRAYED;
    else 
       uEnable =  MF_BYPOSITION | MF_ENABLED;

    EnableMenuItem( hSubMenu, 3, uEnable ); // Acquisition submenu item
    EnableMenuItem( hMenu, 1, uEnable );    // Start menu item
//    EnableMenuItem( hMenu, 2, uEnable );    // Single Value menu item

    //  if NO A/D subsystem...
    //
    if( !config.hAdc ) 
       uEnable = MF_BYPOSITION | MF_GRAYED;
    else  
       uEnable = MF_BYPOSITION | MF_ENABLED;
       
    EnableMenuItem( hSubMenu, 1, uEnable ); // Input submenu item
    EnableMenuItem( hSubMenu, 4, uEnable ); // CGL submenu item

    // if NO D/A subsystem...
    //
    if( !config.hDac ) 
    {
       uEnable = MF_BYPOSITION | MF_GRAYED;
    }   
    else 
    { 
       // enable only if we have WAVEFORM DACs
       olDaGetSSCaps( config.hDac, OLSSC_SUP_CONTINUOUS, &uicap );
       if( uicap )
          uEnable =  MF_BYPOSITION | MF_ENABLED; 
       else
          uEnable =  MF_BYPOSITION | MF_GRAYED;       
    }
    
    EnableMenuItem( hSubMenu, 2, uEnable ); // Output submenu item
    EnableMenuItem( hMenu, 4, uEnable );    // View Output menu item
             
    // C/T subsystem...
    status = olDaGetDevCaps( config.hDev, OLDC_CTELEMENTS, &uicap );

    if( ( status != OLNOERROR ) || ( uicap == 0 ) )
       EnableMenuItem( hMenu, 3, MF_BYPOSITION | MF_GRAYED );
    else 
       EnableMenuItem( hMenu, 3, MF_BYPOSITION | MF_ENABLED );

    DrawMenuBar( hWnd );

}


LRESULT CALLBACK
MainWndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
/*
This is the main message processing handler for the main cexmpl window
*/
{

   ECODE status;
   HMENU hmenu;
   UINT uicap;
   char filemsg[80];
   
    switch( message ) 
    {
        case WM_CREATE:
           config.hwnd = hWnd; 

           if( InitConfig( &config ) ) 
           {
              wsprintf( str, "DT-Open Layers C Example - %s - A/D # %d", (PCHAR)config.BoardName,adelement);
              SetWindowText( hWnd, str );
              
              // Enable/Gray various menu items...
              //
              SetAvailableOptions( hWnd );
			  InitDriveChannelList();
           }
           else 
           {
              // punt if we can't open a board
              MessageBox( hWnd, "Can't open any Open Layers boards.\n"
                                "Check hardware and driver installation",
                                "Error", 
                                MB_ICONEXCLAMATION | MB_OK);
                                
              CloseBoard( &config );
              return -1;
           }
           break;
           
       case WM_SIZE: 
       {
          // calculate where to draw text & graph
           HDC hdc;
           TEXTMETRIC tm;
           int lineht;
           hdc = GetDC( hWnd );
           GetTextMetrics( hdc, &tm );
           ReleaseDC( hWnd, hdc );
           lineht = tm.tmHeight + tm.tmExternalLeading;
           GetClientRect( hWnd, &config.GraphRect );
           config.TextRect = config.GraphRect;
           config.GraphRect.bottom -= 5*lineht/2;  //leave 2.5 lines for text
           
           if( config.GraphRect.bottom < 0 )
              config.GraphRect.bottom = 0;
           
           config.TextRect.top = config.GraphRect.bottom + tm.tmExternalLeading;
           config.uiOldWaveSize = 0;   // clearing window, don't clear old waveform
           InvalidateRect( hWnd, NULL, TRUE);
           break;
       }

       case WM_COMMAND:       /* message: command from application menu */

           switch (LOWORD(wParam)) 
           {
           
              case IDM_ABOUT:
                  DialogBox( hInst, (LPCSTR)ABOUTBOX, hWnd, AboutBox );      
                  break;
                  
              case IDM_BOARD:
                  DialogBox( hInst, (LPCSTR)BOARDBOX, hWnd, BoardBox );
	              wsprintf( str, "DT-Open Layers C Example - %s - A/D # %d", (PCHAR)config.BoardName,adelement);
                  SetWindowText(hWnd,str);
                  // Enable/Gray various menu items...
                  SetAvailableOptions(hWnd);
                  break;
                  
              case IDM_INPUT:
                  DialogBox(hInst, (LPCSTR)IOBOX, hWnd, InputBox );         
                  break;
                  
              case IDM_OUTPUT:
                  DialogBox( hInst, (LPCSTR)IOBOX, hWnd, OutputBox );        
                  break;
                  
              case IDM_ACQ:
                  DialogBox(hInst, (LPCSTR)ACQBOX, hWnd, AcqBox);
                  // disable Start/Stop selection if both A/D & D/A are disabled
                  //
                  if((config.fDisableIn)&&(config.fDisableOut))
                      EnableMenuItem( GetMenu(hWnd), 1, MF_BYPOSITION | MF_GRAYED );
                  else      
                      EnableMenuItem( GetMenu(hWnd), 1, MF_BYPOSITION | MF_ENABLED );
                  DrawMenuBar( hWnd );
                  break;
                  
              case IDM_CGL:
                  DialogBox( hInst, (LPCSTR)CGLBOX, hWnd, CglBox );      
                  break;
                  
              case IDM_ANA_IO:
                  DialogBox( hInst, (LPCSTR)SVBOX, hWnd, SvBox );        
                  break;
                  
              case IDM_DIG_IO:
                  DialogBox( hInst, (LPCSTR)DIGIOBOX, hWnd, DigIOBox );        
                  break;                                     
               
              case IDM_CT:
                  DialogBox( hInst, (LPCSTR)CTBOX, hWnd, CTBox );        
                  break;
                  
              case IDM_VIEW:
                  if( GenerateSignal(&config) == OLNOERROR) 
                  {
                     config.lAdcBufsDone = config.lDacBufsDone = 0;
                     config.fViewDac = TRUE;
                     config.uiOldWaveSize = 0;   // clearing window, don't clear old waveform
                     InvalidateRect( hWnd, NULL, TRUE );
                  }  
                  break;
                  
              case IDM_START:
                  status = StartAcq( &config );
                  if( status  == OLNOERROR) 
                  {
                     hmenu = GetMenu( hWnd );
                     ModifyMenu( hmenu, IDM_START, MF_BYCOMMAND, IDM_STOP, "&Stop!" );
                     EnableMenuItem( hmenu, 0, MF_BYPOSITION | MF_GRAYED );
                     EnableMenuItem( hmenu, 2, MF_BYPOSITION | MF_GRAYED );
                     EnableMenuItem( hmenu, 4, MF_BYPOSITION | MF_GRAYED );
                     DrawMenuBar( hWnd );
                     config.uiOldWaveSize = 0;        // clearing window, don't clear old waveform
                     InvalidateRect( hWnd, NULL, TRUE );  // clear the display
                  }
                  else
                  {
                     SHOW_ERROR( hWnd, status );
                  }   
                  break;
                  
              case IDM_STOP: 
              {
                  hmenu = GetMenu( hWnd );
                  StopAcq( &config );
                  ModifyMenu( GetMenu(hWnd), IDM_STOP,MF_BYCOMMAND, IDM_START, "&Start!" );
                  EnableMenuItem( hmenu, 0, MF_BYPOSITION | MF_ENABLED );
                  EnableMenuItem( hmenu, 2, MF_BYPOSITION | MF_ENABLED );
                  if( config.hDac ) 
                  {
                     // enable only if we have WAVEFORM DACs
                     olDaGetSSCaps( config.hDac, OLSSC_SUP_CONTINUOUS, &uicap );
                     if( uicap )
                        EnableMenuItem(hmenu, 4, MF_BYPOSITION | MF_ENABLED );
                  }
                  DrawMenuBar( hWnd );
                  break;
              }

              default:        // Lets Windows process it       
                  return( DefWindowProc( hWnd, message, wParam, lParam ) );
           }
           break;

       case WM_PAINT:
           PaintGraph( hWnd,&config );
           break;

       case WM_DESTROY:          // window being destroyed 
           CloseBoard( &config );
           PostQuitMessage( 0 );
           break;

       case OLDA_WM_BUFFER_REUSED:
           (*(LPLNG)lParam)++;     // inc buffer done count
           InvalidateRect( hWnd, &config.TextRect, FALSE );
           break;

       case OLDA_WM_BUFFER_DONE:
           (*(LPLNG)lParam)++;     // inc buffer done count
           InvalidateRect( hWnd, &config.TextRect, FALSE );

           if( (HDASS)wParam == config.hAdc ) 
           {          
              // make sure we don't lose the buffer we already have...
              //
              if( config.hAdcBuf ) 
              {
                 if(config.uiDataFlow == IDD_BURST)
                    olDmFreeBuffer(config.hAdcBuf);
                 else
                    olDaPutBuffer(config.hAdc,config.hAdcBuf);
              }
              
              olDaGetBuffer(config.hAdc,&config.hAdcBuf);
              
              //write data to file if requested
              //
              if( config.fFileWrite ) 
              {
                 if( !WriteBufferToFile( hWnd, config.hAdcBuf, filemsg, STRLEN ) ) 
                 {
                    SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
                    MessageBox( hWnd, filemsg, "Error", MB_ICONEXCLAMATION | MB_OK );
                 }
              }
              InvalidateRect( hWnd, NULL, FALSE );  // don't clear graph to avoid "flicker"
           }
           break;
           
       case OLDA_WM_QUEUE_DONE:
           // put app to stopped state  
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           if( IDD_CONTINUOUS ==  config.uiDataFlow )
               MessageBox( hWnd,
                           "Acquisition stopped, rate too fast for current options",
                           "Error", 
                           MB_ICONEXCLAMATION | MB_OK );     
           break;

       case OLDA_WM_TRIGGER_ERROR:
           // put app to stopped state    
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           MessageBox( hWnd, 
                       "Trigger error: acquisition stopped",
                       "Error", 
                       MB_ICONEXCLAMATION | MB_OK );
           break;

       case OLDA_WM_OVERRUN_ERROR:
           // put app to stopped state    
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           MessageBox( hWnd, 
                       "Input overrun error: acquisition stopped",
                       "Error", 
                       MB_ICONEXCLAMATION | MB_OK );
           break;

       case OLDA_WM_UNDERRUN_ERROR:
           // put app to stopped state    
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           MessageBox( hWnd,
                       "Output underrun error: acquisition stopped",
                       "Error", 
                       MB_ICONEXCLAMATION | MB_OK );
           break;

       case OLDA_WM_EVENT_ERROR:
           // put app to stopped state    
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           MessageBox( hWnd,
                       "General Open Layers error: acquisition stopped",
                       "Error", 
                       MB_ICONEXCLAMATION | MB_OK );
           break;

       case OLDA_WM_DEVICE_REMOVAL:
           // put app to stopped state    
           //
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
           MessageBox( hWnd,
                       "General Open Layers error: Device Removed!",
                       "Error", 
                       MB_ICONEXCLAMATION | MB_OK );
		   PostQuitMessage(-1);
           break;

       default:              // Passes it on if unprocessed    
           return( DefWindowProc( hWnd, message, wParam, lParam ) );
    }
    return (0);
}

BOOL CALLBACK
AboutBox( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam )
/*
This function processes messages for "About" dialog box
*/
{
    switch (message) 
    {
       case WM_INITDIALOG: 
          {        
             UINT len = wsprintf( str, "Version " );
             olDaGetVersion( &str[len], STRLEN - len );
             SetDlgItemText( hDlg, IDD_APIVERSION, str );
             len = wsprintf( str, "Driver \"%s\", Version ", (PCHAR)config.BoardName);
             olDaGetDriverVersion( config.hDev, &str[len], STRLEN-len );
             SetDlgItemText( hDlg, IDD_DRVVERSION, str );
             return TRUE;
          }
          break;

       case WM_COMMAND:              
          if( ( IDOK == LOWORD(wParam) ) || ( IDCANCEL == LOWORD(wParam) ) )
          {
             EndDialog( hDlg, TRUE );       
             return TRUE;
          }
          break;
       
       default:
          break;   
    }
    
    return FALSE;               // Didn't process a message    
}

BOOL CALLBACK
BoardBox( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam )
/*
This is the dialog function for the board selection dialog box.  It updates
the board field of the apps CONFIG structure
*/
{
    ECODE ec;

    switch( message ) 
    {
       case WM_INITDIALOG: 
          {
             HWND hwndCombo = GetDlgItem( hDlg, IDD_BOARD );
             ec = olDaEnumBoards( AddBoardList, (LPARAM)hwndCombo );
             wsprintf( str, "%s ", (LPSTR)config.BoardName );
             SendMessage( hwndCombo, CB_SELECTSTRING, 0, (LPARAM)(LPSTR)str );
          }   
          return TRUE;
       
       case WM_COMMAND:
          switch( LOWORD(wParam) ) 
          {
             case IDOK: 
                {
                   int i;
                   char BoardName[MAXBOARDNAME];
                   GetDlgItemText( hDlg, IDD_BOARD, BoardName, MAXBOARDNAME );
                
                   // use only the board name, chop off the extra info
                   for( i=0; i<MAXBOARDNAME-1; i++ ) 
                   {
                      if( BoardName[i] == ' ' )  
                      {
                         BoardName[i] = '\0';
                         break;
                      }
                   }
                   ec = UpdateConfig( &config, BoardName );
                   if( OLNOERROR == ec )
                      EndDialog( hDlg, TRUE );
                   else 
                      SHOW_ERROR( hDlg, ec );
                }
                return TRUE;
             
             case IDCANCEL:
                EndDialog( hDlg, TRUE );
                return TRUE;

             default:
                break;
          }
          break;
          
       default:
          break;
    }
    
    return FALSE;               // Didn't process a message  
}


BOOL CALLBACK 
AddBoardList( LPSTR lpszName, LPSTR lpszEntry, LPARAM lParam )
/*
this is a callback function of olDaEnumBoards, it adds the 
current string the specified combobox (the handle is passed
in lParam)
*/
{
    wsprintf( str, "%s (%s)", lpszName, lpszEntry );
    SendMessage( (HWND)lParam, CB_ADDSTRING, 0, (LPARAM)(LPSTR)str);
    return TRUE;
}

BOOL CALLBACK 
InputBox( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam )
/*
This is the dialog function for the input options dialog box.  If the 
user selects OK, the function updates the current board configuration.
The fScanEnable and dScanFreq fields of the config structure are also
updated.
*/
{
    ECODE status;
    UINT ssdata;

    switch (message) 
    {
       case WM_INITDIALOG:        
          SetWindowText( hDlg, "Input Options" );
          status = InitIOBox( hDlg, config.hAdc, &config, TRUE ); 
          if( status != OLNOERROR) 
          {
             SHOW_ERROR(hDlg,status);
             EndDialog(hDlg, FALSE);
          }
          return TRUE;

       case WM_COMMAND:
          switch (LOWORD(wParam)) 
          {
             case IDOK:
                SaveIOBox( hDlg, config.hAdc, &config);
                EndDialog(hDlg, TRUE);
                return TRUE;
                
             case IDCANCEL:
                EndDialog(hDlg, TRUE);
                return TRUE;

             case IDD_SCANENABLE: 
                {
                   BOOL enable;
                   enable = (BOOL)SendDlgItemMessage( hDlg, 
                                                      IDD_SCANENABLE, 
                                                      BM_GETCHECK, 
                                                      0, 
                                                      0L );
                                                      
                   EnableWindow( GetDlgItem( hDlg, IDD_SCANFREQ ),enable );
                   EnableWindow( GetDlgItem( hDlg, LABEL_IO1 ), enable );
                   return TRUE;
                }
             
             case IDD_SCANFREQ:
                 // when freq edit box is changed, set and read back freq
                 // to display the exact freq that will be used
                if( EN_KILLFOCUS == HIWORD(wParam) ) 
                {    
                   DBL freq;
                   GetWindowText( (HWND)lParam, str, STRLEN );
                   freq = atof(str);
                   status = olDaSetRetriggerFrequency( config.hAdc, freq );
                   
                   if( OLNOERROR == status )
                       status = olDaConfig(config.hAdc);
                       
                   if( OLNOERROR != status )
                       SHOW_ERROR( hDlg, status );
                       
                   olDaGetRetriggerFrequency( config.hAdc, &freq );
                   sprintf( str, FREQ_FORMAT, freq );
                   SetWindowText( (HWND)lParam, str );
                   return TRUE;
                 }
              case IDD_RANGE:
              {
                switch(HIWORD(wParam))
                    case CBN_SELCHANGE:
                    {
                    DBL min,max;
                    // Update Encoding...
                    //range
                    GetDlgItemText( hDlg, IDD_RANGE, str, STRLEN );
                    sscanf( str, "%lfV to %lf", &min, &max );
                    olDaSetRange( config.hAdc, max, min );
                    olDaConfig(config.hAdc);
                    CHECKERROR( olDaGetEncoding( config.hAdc, &ssdata ) );
                    if( ssdata == OL_ENC_BINARY )
                        {
                        SendDlgItemMessage( hDlg, IDD_BINARY, BM_SETCHECK, 1, 0L );
                        SendDlgItemMessage( hDlg, IDD_2SCOMP, BM_SETCHECK, 0, 0L );
                        }
                    else
                        {
                        SendDlgItemMessage( hDlg, IDD_2SCOMP, BM_SETCHECK, 1, 0L );
                        SendDlgItemMessage( hDlg, IDD_BINARY, BM_SETCHECK, 0, 0L );
                        }
                    }
               }
             default:
                 break ;  
          }
          break;

       default:
          break ;  
          
    }
    return FALSE;
}

BOOL CALLBACK 
OutputBox( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam )
/*
This is the dialog function for the output options dialog box.  If the 
user selects OK, the function updates the current board configuration.
*/
{
    ECODE status;
    switch (message) 
    {
       case WM_INITDIALOG:        /* message: initialize dialog box */
           SetWindowText(hDlg,"Output Options");
           status = InitIOBox(hDlg,config.hDac,&config,FALSE); 
           if( status != OLNOERROR) 
           {
              SHOW_ERROR(hDlg,status);
              EndDialog(hDlg, FALSE);
           }
           return TRUE;

       case WM_COMMAND:
          switch (LOWORD(wParam)) 
          {
             case IDOK:
                SaveIOBox(hDlg,config.hDac,&config);
                EndDialog(hDlg, TRUE);
                return TRUE;
             
             case IDCANCEL:
                EndDialog(hDlg, TRUE);
                return TRUE;
             
             default:
                break ;  
          }
          break ;  

       default:
          break ;  
    }
    return FALSE;
}

ECODE 
InitIOBox( HWND hDlg, HDASS hDass, LPCONFIG lpconfig, BOOL fShowScan )
/*
This function sets the initial state of both the input and output
options dialog box.  It sets the box to match the current settings
of the specified subsystem.  The function also greys any options
that the subsystem does not support.  If the fShowScan parameter is
FALSE, the function will hide all the fields related to triggered
scan.
*/
{
    UINT ssdata;
    DBL min,max;

    //interface mode
    CHECKERROR( olDaGetChannelType( hDass, &ssdata ) );
    if( ssdata == OL_CHNT_SINGLEENDED )
       SendDlgItemMessage( hDlg, IDD_SINGLEENDED, BM_SETCHECK, 1, 0L );
    else
       SendDlgItemMessage( hDlg, IDD_DIFFERENTIAL, BM_SETCHECK, 1, 0L );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_SINGLEENDED, &ssdata ) );
    if( !ssdata )
        EnableWindow( GetDlgItem( hDlg, IDD_SINGLEENDED), FALSE );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_DIFFERENTIAL, &ssdata ) );
    if( !ssdata )
        EnableWindow( GetDlgItem( hDlg, IDD_DIFFERENTIAL ), FALSE );
    
    // Clock Info
    CHECKERROR( olDaGetClockSource( hDass, &ssdata ) );
    if( ssdata == OL_CLK_INTERNAL )
        SendDlgItemMessage( hDlg, IDD_CLK_INTERNAL, BM_SETCHECK, 1, 0L );
    else
        SendDlgItemMessage( hDlg, IDD_CLK_EXTERNAL, BM_SETCHECK, 1, 0L );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_INTCLOCK, &ssdata ) );
    if( !ssdata )
        EnableWindow( GetDlgItem( hDlg, IDD_CLK_INTERNAL), FALSE );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_EXTCLOCK, &ssdata ) );
    if( !ssdata )
        EnableWindow(GetDlgItem(hDlg,IDD_CLK_EXTERNAL),FALSE);

    //trigger source
    CHECKERROR( olDaGetTrigger( hDass, &ssdata ) );
    if( ssdata == OL_TRG_DIGITALEVENT )
        SendDlgItemMessage(hDlg,IDD_TRG_DIGITAL,BM_SETCHECK,1,0L);
    if( ssdata == OL_TRG_SOFT )
        SendDlgItemMessage(hDlg,IDD_TRG_INTERNAL,BM_SETCHECK,1,0L);
    if( ssdata == OL_TRG_EXTERN )
        SendDlgItemMessage(hDlg,IDD_TRG_EXTERNAL,BM_SETCHECK,1,0L);

    CHECKERROR( olDaGetSSCaps(hDass,OLSSC_SUP_SOFTTRIG,&ssdata) );
    if( !ssdata )
       EnableWindow(GetDlgItem( hDlg, IDD_TRG_INTERNAL ), FALSE );
    
    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_EXTERNTRIG, &ssdata ) );
    if( !ssdata )
        EnableWindow(GetDlgItem(hDlg,IDD_TRG_EXTERNAL),FALSE);

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_DIGITALEVENTTRIG, &ssdata ) );
    if( !ssdata )
        EnableWindow(GetDlgItem(hDlg,IDD_TRG_DIGITAL),FALSE);

    //range
    CHECKERROR( olDaEnumSSCaps( hDass, 
                                OL_ENUM_RANGES, 
                                AddRangeList, 
                                (LPARAM)GetDlgItem( hDlg, IDD_RANGE ) ) );

    CHECKERROR( olDaGetRange( hDass, &max, &min ) );

    sprintf( str, RANGE_FORMAT, min,max );
    SendDlgItemMessage( hDlg, IDD_RANGE, CB_SELECTSTRING, (WPARAM)-1, (LPARAM)(LPSTR)str );

    //encoding
    CHECKERROR( olDaGetEncoding( hDass, &ssdata ) );
    if( ssdata == OL_ENC_BINARY )
        SendDlgItemMessage( hDlg, IDD_BINARY, BM_SETCHECK, 1, 0L );
    else
        SendDlgItemMessage( hDlg, IDD_2SCOMP, BM_SETCHECK, 1, 0L );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_BINARY, &ssdata ) );
    if( !ssdata )
        EnableWindow( GetDlgItem( hDlg, IDD_BINARY ), FALSE );

    CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_2SCOMP, &ssdata ) );
    if( !ssdata )
        EnableWindow( GetDlgItem( hDlg, IDD_2SCOMP ), FALSE  );

    //triggered scan
    if( fShowScan  ) 
    {
       CHECKERROR( olDaGetSSCaps( hDass, OLSSC_SUP_TRIGSCAN, &ssdata ) );
       if( !ssdata  ) 
       {
          EnableWindow( GetDlgItem( hDlg, IDD_SCANENABLE ), FALSE );
          EnableWindow( GetDlgItem( hDlg, IDD_SCANFREQ ), FALSE );
          EnableWindow( GetDlgItem( hDlg, LABEL_IO1 ), FALSE );
          EnableWindow( GetDlgItem( hDlg, LABEL_IO2 ), FALSE );
       }
       else 
       {
          sprintf( str, FREQ_FORMAT, lpconfig->dScanFreq );
          SendDlgItemMessage( hDlg,
                              IDD_SCANENABLE,BM_SETCHECK,
                              ( WPARAM )lpconfig->fScanEnable,
                              0L );
                              
          SetDlgItemText( hDlg, 
                          IDD_SCANFREQ, 
                          str );
                          
          if( !lpconfig->fScanEnable ) 
          {
             EnableWindow( GetDlgItem( hDlg, IDD_SCANFREQ ), FALSE );
             EnableWindow( GetDlgItem( hDlg, LABEL_IO1 ), FALSE );
          }
       }
    }
    else 
    {
       ShowWindow( GetDlgItem( hDlg, IDD_SCANENABLE ), SW_HIDE );
       ShowWindow( GetDlgItem( hDlg, IDD_SCANFREQ ), SW_HIDE );
       ShowWindow( GetDlgItem( hDlg, LABEL_IO1 ), SW_HIDE );
       ShowWindow( GetDlgItem( hDlg, LABEL_IO2 ), SW_HIDE );
    }

    return OLNOERROR;
}


BOOL CALLBACK 
AddRangeList( UINT cap,  DBL max,  DBL min,  LPARAM lParam )
/*
This is a callback function of olDaEnumSSCaps,  it adds the
the specified range to the specified combobox ( the handle is passed
in lParam )
*/
{
    sprintf( str, RANGE_FORMAT, min, max );
    SendMessage( ( HWND )lParam, CB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
    return TRUE ;
}

VOID 
SaveIOBox( HWND hDlg,  HDASS hDass,  LPCONFIG lpconfig )
/*
This function updates the board configuration based on the current state
of the dialog controls
*/
{
    DBL min,  max;

    //interface mode
    if( SendDlgItemMessage( hDlg, IDD_SINGLEENDED, BM_GETCHECK, 0, 0L ) )
        olDaSetChannelType( hDass, OL_CHNT_SINGLEENDED );
    else
        olDaSetChannelType( hDass, OL_CHNT_DIFFERENTIAL );
    
    //clock source
    if( SendDlgItemMessage( hDlg, IDD_CLK_INTERNAL, BM_GETCHECK, 0, 0L ) )
        olDaSetClockSource( hDass, OL_CLK_INTERNAL );
    else
        olDaSetClockSource( hDass, OL_CLK_EXTERNAL );

    //trigger source
    if( SendDlgItemMessage( hDlg, IDD_TRG_DIGITAL, BM_GETCHECK, 0, 0L ) )
        olDaSetTrigger( hDass, OL_TRG_DIGITALEVENT );
    else
    {
        if( SendDlgItemMessage( hDlg, IDD_TRG_INTERNAL, BM_GETCHECK, 0, 0L ) )
            olDaSetTrigger( hDass, OL_TRG_SOFT );
        else
            olDaSetTrigger( hDass, OL_TRG_EXTERN );
    }

    //range
    GetDlgItemText( hDlg, IDD_RANGE, str, STRLEN );
    sscanf( str, "%lfV to %lf", &min, &max );
    olDaSetRange( hDass, max, min );

    //encoding
    if( SendDlgItemMessage( hDlg, IDD_BINARY, BM_GETCHECK, 0, 0L ) )
        olDaSetEncoding( hDass, OL_ENC_BINARY );
    else
        olDaSetEncoding( hDass, OL_ENC_2SCOMP );

    //triggered scan
    if( IsWindowVisible( GetDlgItem( hDlg, IDD_SCANENABLE ) )  ) {
        lpconfig->fScanEnable = ( BOOL )SendDlgItemMessage( hDlg, IDD_SCANENABLE, BM_GETCHECK, 0, 0L );
        GetDlgItemText( hDlg, IDD_SCANFREQ, str, STRLEN );
        lpconfig->dScanFreq = atof( str );
    }
}
    
BOOL CALLBACK 
AcqBox( HWND hDlg,  UINT message,  WPARAM wParam,  LPARAM lParam )
/*
This is the dialog function for the Acquisition options dialog box.  It
updates the apps CONFIG structure based on user input.
*/
{
    DBL freq;
    DBL dmin, dmax;
    ECODE status;
    UINT uicap;
    
    
    switch ( message ) {
    case WM_INITDIALOG: {      /* message: initialize dialog box */
        SetClassLong( hDlg, GCL_HBRBACKGROUND, ( long )( HBRUSH )CreateSolidBrush( ( COLORREF )RGB( 192, 192, 192 ) )  );
        LoadAcqBox( hDlg, &config );
       if( !config.hAdc )
          EnableWindow( GetDlgItem( hDlg, IDD_DISABLEIN ), FALSE );  
          
        if( !config.hDac )
            EnableWindow( GetDlgItem( hDlg, IDD_DISABLEOUT ), FALSE );
        else {
                // disable control if we don't have WAVEFORM DACs
                olDaGetSSCaps( config.hDac, OLSSC_SUP_CONTINUOUS, &uicap );
                if( uicap != TRUE )
                    EnableWindow( GetDlgItem( hDlg, IDD_DISABLEOUT ), FALSE );
        }
        return TRUE;
    }

    case WM_COMMAND:

        switch ( LOWORD( wParam ) ) {

        case IDOK:
            SaveAcqBox( hDlg, &config );
            EndDialog( hDlg,  TRUE );
            return TRUE;
        case IDCANCEL:
            // restore output buffer size only if we have WAVEFORM DACS
            olDaGetSSCaps( config.hDac, OLSSC_SUP_CONTINUOUS, &uicap );
            if( uicap == TRUE )
                SetWaveFreq( config.dWaveFreq, config.dOutFreq, &config.lOutBufSize );  
            EndDialog( hDlg,  TRUE );
            return TRUE;
        case IDD_DISABLEIN: {
            BOOL enable;
            enable = !( BOOL )SendDlgItemMessage( hDlg, IDD_DISABLEIN, BM_GETCHECK, 0, 0L );
            EnableWindow( GetDlgItem( hDlg, IDD_DISPLAY    ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_FILEWRITE  ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_BURST      ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_CONTINUOUS ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_INFREQ     ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_NUMBUF     ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_BUFSIZE    ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_INDMA      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN1      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN2      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN3      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN4      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN5      ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_IN6      ), enable );
            return TRUE;
        }
        case IDD_DISABLEOUT: {
            BOOL enable;
            enable = !( BOOL )SendDlgItemMessage( hDlg, IDD_DISABLEOUT, BM_GETCHECK, 0, 0L );
            EnableWindow( GetDlgItem( hDlg, LABEL_OUT1  ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_OUT2  ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_OUT3  ), enable );
            EnableWindow( GetDlgItem( hDlg, LABEL_OUT4   ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_PEAK     ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_WAVEFREQ ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_OUTFREQ  ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_OUTDMA   ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_SINE     ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_SQUARE   ), enable );
            EnableWindow( GetDlgItem( hDlg, IDD_TRIANGLE ), enable );
            return TRUE;
        }
        case IDD_INFREQ:
            // when freq edit box is changed,  set and read back freq
            // to display the exact freq that will be used

             if( HIWORD( wParam )==EN_KILLFOCUS ) {
                GetWindowText( ( HWND )lParam, str, STRLEN );

                freq = atof( str );
                status = olDaSetClockFrequency( config.hAdc, freq );
                if( status == OLNOERROR )
                    status = olDaConfig( config.hAdc );
                if( status != OLNOERROR )
                    SHOW_ERROR( hDlg, status );
                olDaGetClockFrequency( config.hAdc, &freq );
                sprintf( str, FREQ_FORMAT, freq );
                SetWindowText( ( HWND )lParam, str );
                return TRUE;
            }
            break;
        case IDD_OUTFREQ:
            // when freq edit box is changed,  set and read back freq
            // to display the exact freq that will be used

           if( HIWORD( wParam )==EN_KILLFOCUS ) {
                GetWindowText( ( HWND )lParam, str, STRLEN );
                
                freq = atof( str );
                status = olDaSetClockFrequency( config.hDac, freq );
                if( status == OLNOERROR )
                    status = olDaConfig( config.hAdc );
                if( status != OLNOERROR )
                    SHOW_ERROR( hDlg, status );
                olDaGetClockFrequency( config.hDac, &freq );
                sprintf( str, FREQ_FORMAT, freq );
                SetWindowText( ( HWND )lParam, str );
                PostMessage( hDlg, WM_COMMAND, IDD_WAVEFREQ, 
                            MAKELPARAM( GetDlgItem( hDlg, IDD_WAVEFREQ ), EN_KILLFOCUS ) );
                return TRUE;
            }
            break;
        case IDD_WAVEFREQ:
            // calculate output buffer size and display exact wave freq


           if( HIWORD( wParam )==EN_KILLFOCUS ) {

               DBL SampFreq;
                GetWindowText( ( HWND )lParam, str, STRLEN );
                freq = atof( str );
                GetDlgItemText( hDlg, IDD_OUTFREQ, str, STRLEN );
                SampFreq = atof( str );
                freq = SetWaveFreq( freq, SampFreq, &config.lOutBufSize );
                sprintf( str, FREQ_FORMAT, freq );
                SetWindowText( ( HWND )lParam, str );
                return TRUE;
            }
            break;
        case IDD_PEAK:

           if( HIWORD( wParam )==EN_KILLFOCUS ) {
                GetWindowText( ( HWND )lParam, str, STRLEN );

                config.dPeakVolts = atof( str );
                status = olDaGetRange( config.hDac, &dmax, &dmin );
                if( config.dPeakVolts > dmax ) config.dPeakVolts = dmax;
                if( config.dPeakVolts < dmin ) config.dPeakVolts = dmin;
                
                sprintf( str, FREQ_FORMAT, config.dPeakVolts );
                SetWindowText( ( HWND )lParam, str );

            }
            break;        

        default:
            break;
        }
    }
    return FALSE;               /* Didn't process a message    */
}

BOOL 
InitConfig( LPCONFIG lpconfig )
/*
This function initializes the specified configuration data structure.  It
opens the first board that can be configured without errors.  If no board can
be opened,  the function returns FALSE.  Otherwise,  it returns TRUE.
*/
{

    // init config structure = no current board
    lpconfig->hDev = NULL;
    lpconfig->hAdc = lpconfig->hDac = lpconfig->hCT0 = NULL;
    lpconfig->BoardName[0] = '\0';
    lpconfig->bUseWinMsgs = TRUE;

    olDaEnumBoards( InitBoard, ( LPARAM )lpconfig );
    return ( lpconfig->hDev != 0 );       // return TRUE ( success ) if a board was opened
}


BOOL CALLBACK 
InitBoard( LPSTR lpszName,  LPSTR lpszEntry,  LPARAM lpconfig )
/*
This is a callback function of olDaEnumBoards,  it tries to open the
specified board.  The function returns FALSE if successful to stop 
enumeration.  Otherwise,  the function returns TRUE to try the next board
*/
{
    if( UpdateConfig( ( LPCONFIG )lpconfig,  lpszName ) == OLNOERROR )
    {
       return FALSE;      // stop enumeration if we can open current board
    }    
    else 
    {
       return TRUE;       // otherwise,  continue enumeration to try next board
    }
}


ECODE 
UpdateConfig( LPCONFIG lpconfig, LPSTR lpszBoardName )
/*
This function updates the specified configuration based on the
input board name.  If the name is the same as the current board, 
the routine does nothing.  Otherwise,  the function closes the
old board ( if any ),  opens the new board,  and inits the
configuration.  The function returns OLNOERROR is successful.
Otherwise,  the function returns the appropriate error code.
*/
{
    ECODE status;
    UINT cap;

    if( lstrcmp( lpconfig->BoardName, lpszBoardName ) == 0 )
        return OLNOERROR;

    CloseBoard( lpconfig );

    // update board name
    lpconfig->BoardName[MAXBOARDNAME-1] = '\0';
    lstrcpyn( lpconfig->BoardName, lpszBoardName, MAXBOARDNAME-1 );
    
    // open subsystems
    status = olDaInitialize( lpconfig->BoardName, &lpconfig->hDev );
    if( status != OLNOERROR ) 
       return status;
    
    // A/D Subsystem setup
    //
    status = olDaGetDASS( lpconfig->hDev, OLSS_AD, adelement, &lpconfig->hAdc );
    if( status == OLNOERROR )
    {
       lpconfig->fDisableIn = FALSE;     
       // set the hwnd for return messages
       CHECKERROR( olDaSetWndHandle( lpconfig->hAdc, 
                                     lpconfig->hwnd, 
                                     ( LPARAM )( LPLNG )&lpconfig->lAdcBufsDone ) );

       // init sample frequency hardware
       CHECKERROR( olDaSetClockFrequency( lpconfig->hAdc, 1000000.0 ) );

       // read back frequency and update struct with exact values
       CHECKERROR( olDaConfig( lpconfig->hAdc ) );
       CHECKERROR( olDaGetClockFrequency( lpconfig->hAdc, &lpconfig->dInFreq ) );

       // init  retrigger freq if supported
       CHECKERROR( olDaGetSSCaps( lpconfig->hAdc, OLSSC_SUP_TRIGSCAN, &cap ) );
       if( cap ) 
       {
          CHECKERROR( olDaSetRetriggerFrequency( lpconfig->hAdc, 10.0 ) );
          CHECKERROR( olDaConfig( lpconfig->hAdc ) );
          CHECKERROR( olDaGetRetriggerFrequency( lpconfig->hAdc, &lpconfig->dScanFreq ) );
       }

       CHECKERROR( olDaGetSSCaps( lpconfig->hAdc, OLSSC_NUMDMACHANS, &cap ) );
       if( cap > 1 ) cap=1;
       if( olDaSetDmaUsage( lpconfig->hAdc, cap ) )
          lpconfig->uiInDma = 0;
       else 
          lpconfig->uiInDma = cap;
    }
    else 
    {
       lpconfig->hAdc = NULL;
       config.fDisableIn = TRUE;
    }
    
    // D/A subsystem setup
    //
    if( OLNOERROR == ( status = olDaGetDASS( lpconfig->hDev, OLSS_DA, 0, &lpconfig->hDac ) ) )
    {
       lpconfig->fDisableOut = TRUE;  // Start out with DAC disabled  
      
       olDaSetChannelType( lpconfig->hDac, OL_CHNT_DIFFERENTIAL );
      
       CHECKERROR( olDaGetSSCaps( lpconfig->hDac, OLSSC_SUP_CONTINUOUS, &cap ) );
       if( cap ) 
       {
          // set the hwnd for return messages
          CHECKERROR( olDaSetWndHandle( lpconfig->hDac, 
                                        lpconfig->hwnd, 
                                        ( LPARAM )( LPLNG )&lpconfig->lDacBufsDone ) );
          // init sample frequency hardware
          CHECKERROR( olDaSetClockFrequency( lpconfig->hDac, 1000.0 ) );
                        
          // read back frequency and update struct with exact values
          CHECKERROR( olDaConfig( lpconfig->hDac ) );
          CHECKERROR( olDaGetClockFrequency( lpconfig->hDac, &lpconfig->dOutFreq ) );
    
          // init  retrigger freq if supported
          CHECKERROR( olDaGetSSCaps( lpconfig->hDac, OLSSC_SUP_TRIGSCAN, &cap ) );
          if( cap ) 
          {
             CHECKERROR( olDaSetRetriggerFrequency( lpconfig->hDac, 10.0 ) );
             CHECKERROR( olDaConfig( lpconfig->hDac ) );
             CHECKERROR( olDaGetRetriggerFrequency( lpconfig->hDac, &lpconfig->dScanFreq ) );
          }
          CHECKERROR( olDaGetSSCaps( lpconfig->hDac, OLSSC_NUMDMACHANS, &cap ) );
          if( cap > 1 ) cap=1;
          if( olDaSetDmaUsage( lpconfig->hDac, cap ) )
             lpconfig->uiInDma = 0;
          else 
             lpconfig->uiInDma = cap;
           
        }// end if we have WAVEFORM DACS
   }
   else
   {
      lpconfig->hDac=NULL;
      config.fDisableOut = TRUE;
   }

   // init remaining config parameters
   lpconfig->hCT0 = NULL;
   lpconfig->fScanEnable = FALSE;

   lpconfig->fDisplay = FALSE;
   lpconfig->fFileWrite = TRUE;
   lpconfig->uiDataFlow = IDD_CONTINUOUS;
   lpconfig->uiNumBuf = 3;
   lpconfig->ulBufSize = 131072;

   lpconfig->uiOutDma = 0;
   lpconfig->uiWaveType = IDD_SINE;  
   lpconfig->dPeakVolts = 1.0; 
   
   olDaGetSSCaps( lpconfig->hDac, OLSSC_SUP_CONTINUOUS, &cap );
   if( cap )
   { 
      lpconfig->dWaveFreq = SetWaveFreq( 60.0, lpconfig->dOutFreq, &lpconfig->lOutBufSize );
   }    
   else 
   {                          
      // just set some defaults
      lpconfig->dOutFreq = 1.0;
      lpconfig->dWaveFreq = 1.0;
      lpconfig->lOutBufSize = 10;
   }
   
   lpconfig->hDacBuf = NULL;
   lpconfig->hAdcBuf = NULL;
   lpconfig->fViewDac = FALSE;
   lpconfig->lAdcBufsDone = 0;
   lpconfig->lDacBufsDone = 0;

   return OLNOERROR;
}

DBL 
SetWaveFreq( DBL dDesiredFreq,  DBL dSampFreq,  LPLNG lplBufSize )
/*
This function sets the output waveform frequency. The function computes
the output buffer size so that there will be minimal phase discontinuity when
the buffer is sent continuously in wrap single mode. The routine returns the
the wave frequency that will cause no discontinuity.
*/
{
    DBL freq,  normfreq;
    long reps;
    DBL phaseErr,  minerr;
    long i;
    DBL phase;  //phase in cycles

    freq = dDesiredFreq;

    //check for illegal frequency
    if( freq > dSampFreq / 2  ) 
    {
       *lplBufSize = MAXOUTPUTSIZE/2;     
       return ( dSampFreq / 2 );
    }

    //if phase error is < 1% don't bother to compute new buffer size
    //
    if( *lplBufSize <= MAXOUTPUTSIZE && *lplBufSize >= MAXOUTPUTSIZE/2 ) 
    {
       phase = *lplBufSize * freq / dSampFreq;
       phaseErr = fabs( phase - ( long )( phase + .5 ) );
       if( phaseErr < .01 )
           return freq;
    }

    //select buffer size for continuous phase
    // buffer size = repeats * sample freq / freq

    normfreq = freq / dSampFreq;   //normalized frequency
    minerr = 1;
    for ( i = MAXOUTPUTSIZE/2; i <= MAXOUTPUTSIZE; i++ ) 
    {
       reps = ( long )( i * normfreq + .5 );  //round to nearest whole repeats
       phaseErr = fabs( i * normfreq - reps );
       if( phaseErr < minerr ) 
       {
          minerr = phaseErr;
          *lplBufSize = i;
       }
    }

    //tweak the freq for continuous phase
    reps = ( long )( *lplBufSize * normfreq + .5 );
    freq = dSampFreq * reps / *lplBufSize;
    return freq;
}


VOID 
LoadAcqBox( HWND hDlg, LPCONFIG lpconfig )
/*
This function is used to init the controls of the acquisition options
dialog box
*/
{
    UINT dma,  i,  uicap;

    // If No Input Subsystem,  then enable Output subsystem
    if( !lpconfig->hAdc )
    {
       lpconfig->fDisableIn = 1;
       if( lpconfig->hDac ) lpconfig->fDisableOut=0;
    } 
   
    // if no WAVEFORM DACS,  disable output
    if( !lpconfig->hDac ) 
    {
       lpconfig->fDisableOut=1;
    }
    else 
    {
       olDaGetSSCaps( lpconfig->hDac, OLSSC_SUP_CONTINUOUS, &uicap );
       if( uicap != TRUE ) lpconfig->fDisableOut=1; 
    }
    
    // Set Input Option Settings
    SendDlgItemMessage( hDlg, IDD_DISABLEIN, BM_SETCHECK, lpconfig->fDisableIn, 0L );
    SendDlgItemMessage( hDlg, IDD_DISPLAY, BM_SETCHECK, lpconfig->fDisplay, 0L );

    SendDlgItemMessage( hDlg, IDD_FILEWRITE, BM_SETCHECK, lpconfig->fFileWrite, 0L );
    SendDlgItemMessage( hDlg, lpconfig->uiDataFlow, BM_SETCHECK, 1, 0L );

    sprintf( str, FREQ_FORMAT, lpconfig->dInFreq );  
    SetDlgItemText( hDlg, IDD_INFREQ, str );

    sprintf( str, "%d", lpconfig->uiNumBuf );
    SetDlgItemText( hDlg, IDD_NUMBUF, str );
    sprintf( str, "%ld", lpconfig->ulBufSize );
    SetDlgItemText( hDlg, IDD_BUFSIZE, str );

    dma = 0;
    olDaGetSSCaps( lpconfig->hAdc, OLSSC_NUMDMACHANS, &dma );
    for ( i=0; i <= dma; i++ ) 
    {
       sprintf( str, "%d", i );
       SendDlgItemMessage( hDlg, IDD_INDMA, CB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
    }
    SendDlgItemMessage( hDlg, IDD_INDMA, CB_SETCURSEL, lpconfig->uiInDma, 0L );

    PostMessage( hDlg, WM_COMMAND, IDD_DISABLEIN, 0L );    //process the checkbox
    
    // Set Output Option Settings
    SendDlgItemMessage( hDlg, IDD_DISABLEOUT, BM_SETCHECK, lpconfig->fDisableOut, 0L );
    SendDlgItemMessage( hDlg, IDD_OUTDMA, BM_SETCHECK, lpconfig->uiOutDma, 0L );
    SendDlgItemMessage( hDlg, lpconfig->uiWaveType, BM_SETCHECK, 1, 0L );
        
    sprintf( str, FREQ_FORMAT, lpconfig->dPeakVolts );
    SetDlgItemText( hDlg, IDD_PEAK, str );
    sprintf( str, FREQ_FORMAT, lpconfig->dWaveFreq );
    SetDlgItemText( hDlg, IDD_WAVEFREQ, str );
    sprintf( str, FREQ_FORMAT, lpconfig->dOutFreq );
    SetDlgItemText( hDlg, IDD_OUTFREQ, str );
        
    PostMessage( hDlg, WM_COMMAND, IDD_DISABLEOUT, 0L );   //process the checkbox
}

VOID 
SaveAcqBox( HWND hDlg, LPCONFIG lpconfig )
/*
This function updates the CONFIG structure based on the current state
of the dialog controls
*/
{
    lpconfig->fDisableIn = ( BOOL )SendDlgItemMessage( hDlg, IDD_DISABLEIN, BM_GETCHECK, 0, 0L );
    lpconfig->fDisplay = ( BOOL )SendDlgItemMessage( hDlg, IDD_DISPLAY, BM_GETCHECK, 0, 0L );
    lpconfig->fFileWrite = ( BOOL )SendDlgItemMessage( hDlg, IDD_FILEWRITE, BM_GETCHECK, 0, 0L );
    if( SendDlgItemMessage( hDlg, IDD_BURST, BM_GETCHECK, 0, 0L ) )
        lpconfig->uiDataFlow = IDD_BURST;
    else
        lpconfig->uiDataFlow = IDD_CONTINUOUS;

    GetDlgItemText( hDlg, IDD_INFREQ, str, STRLEN );
    lpconfig->dInFreq = atof( str );
    GetDlgItemText( hDlg, IDD_NUMBUF, str, STRLEN );
    lpconfig->uiNumBuf = atoi( str );
    GetDlgItemText( hDlg, IDD_BUFSIZE, str, STRLEN );
    lpconfig->ulBufSize = atol( str );

    lpconfig->uiInDma = ( UINT )SendDlgItemMessage( hDlg, IDD_INDMA, CB_GETCURSEL, 0, 0L );

    lpconfig->fDisableOut = ( BOOL )SendDlgItemMessage( hDlg, IDD_DISABLEOUT, BM_GETCHECK, 0, 0L );
    // output dma can only be 0 or 1
    lpconfig->uiOutDma = ( UINT )SendDlgItemMessage( hDlg, IDD_OUTDMA, BM_GETCHECK, 0, 0L );

    if( SendDlgItemMessage( hDlg, IDD_SINE, BM_GETCHECK, 0, 0L ) )
        lpconfig->uiWaveType = IDD_SINE;
    else if( SendDlgItemMessage( hDlg, IDD_SQUARE, BM_GETCHECK, 0, 0L ) )
        lpconfig->uiWaveType = IDD_SQUARE;
    else
        lpconfig->uiWaveType = IDD_TRIANGLE;
    
    GetDlgItemText( hDlg, IDD_PEAK, str, STRLEN );
    lpconfig->dPeakVolts = atof( str );
    GetDlgItemText( hDlg, IDD_WAVEFREQ, str, STRLEN );
    lpconfig->dWaveFreq = atof( str );
    GetDlgItemText( hDlg, IDD_OUTFREQ, str, STRLEN );
    lpconfig->dOutFreq = atof( str );
        
}

VOID InitDriveChannelList()
{
    UINT i,maxch,setting;
	UINT ListSize;
	UINT MaxCglSize;

    olDaGetSSCaps( config.hAdc, OLSSC_CGLDEPTH, &MaxCglSize );
    MaxCglSize = min( MAXCGLSIZE, MaxCglSize );
	olDaGetChannelType(config.hAdc,&setting);
    if(setting=OL_CHNT_SINGLEENDED)
		olDaGetSSCaps(config.hAdc,OLSSC_MAXSECHANS,&maxch);
    else
		olDaGetSSCaps(config.hAdc,OLSSC_MAXDICHANS,&maxch);

    // set up 8 entries: channel 0-6,32;  gain 1.0
	for (i = 0; i < 8 && i < maxch; i++) {
		olDaSetChannelListEntry( config.hAdc, i, i==7? 32 : i );
		olDaSetGainListEntry( config.hAdc, i, 1.0 );
	}

    for(i=0;i<maxch;i++)
		olDaSetChannelFilter( config.hAdc, i, 0.0 );

	//change list size
    ListSize = min( 8, MaxCglSize );
    olDaSetChannelListSize( config.hAdc, ListSize );
}

BOOL CALLBACK 
CglBox( HWND hDlg,  UINT message,  WPARAM wParam,  LPARAM lParam )
/*
This is the dialog function for the channel gain list setup dialog box.  It
updates the subsystem's channel list,  gain list ,  and filter settings
based on user input.
*/
{
    static UINT ListSize;
    static UINT MaxCglSize;
    static UINT SelectIndex;
    UINT i;
    ECODE status;

    switch ( message ) 
    {

       case WM_INITDIALOG:        /* message: initialize dialog box */
           SelectIndex = 0;
           status = olDaGetChannelListSize( config.hAdc, &ListSize );
           if(  status != OLNOERROR ) 
           {
              SHOW_ERROR( hDlg, status );
              EndDialog( hDlg,  FALSE );
              return TRUE;
           }
           olDaGetSSCaps( config.hAdc, OLSSC_CGLDEPTH, &MaxCglSize );
           MaxCglSize = min( MAXCGLSIZE, MaxCglSize );

           // init list box
           i = 36;     // set tab stops every 9 characters ( average )

           SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_SETTABSTOPS, 1, ( LPARAM )( int* )&i );
           for( i=0; i < ListSize; i++ ) 
           {
              GetCGLString( &config, i, str );
              SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
           }

           // init CGL size
           wsprintf( str, "%d", ListSize );
           SetDlgItemText( hDlg, IDD_CGLSIZE, str );

           // init unused portion of CGL ( in case user increases ListSize )
           for ( i=ListSize; i<MaxCglSize; i++ ) {
               olDaSetChannelListEntry( config.hAdc, i, 0 );
               olDaSetGainListEntry( config.hAdc, i, 1.0 );
           }

           // init channel combo box
           InitChannelComboBox( GetDlgItem( hDlg, IDD_CGLCHANNEL ), config.hAdc );

           // init gain combo box
           olDaEnumSSCaps( config.hAdc, OL_ENUM_GAINS, AddGainList, 
                       ( LPARAM )GetDlgItem( hDlg, IDD_CGLGAIN )  );
           SendDlgItemMessage( hDlg, IDD_CGLGAIN, CB_SETCURSEL, 0, 0L );

           // init entry
           SetDlgItemText( hDlg, IDD_CGLENTRY, "0" );

           // init filter combo box
           olDaEnumSSCaps( config.hAdc, OL_ENUM_FILTERS, AddFilterList, 
                       ( LPARAM )GetDlgItem( hDlg, IDD_CGLFILTER )  );
           SendDlgItemMessage( hDlg, IDD_CGLFILTER, CB_SETCURSEL, 0, 0L );


           return TRUE;

       case WM_COMMAND:
           switch ( LOWORD( wParam ) ) 
           {
              case IDCANCEL: 
                 EndDialog( hDlg,  TRUE );
                 return TRUE;
           
              case IDOK: 
              {
                  char errstr[250];
                  status = olDaConfig( config.hAdc );
                  if( status != OLNOERROR ) 
                  {
                     olDaGetErrorString( status, str, STRLEN );
                     wsprintf( errstr, "%s.  Your CGL is not supported by the current hardware.  "
                                     "Please correct or reset the CGL.", str );
                     MessageBox( hDlg, errstr, "Error",  MB_ICONEXCLAMATION | MB_OK );
                  }
                  else
                  {
                     EndDialog( hDlg,  TRUE );
                  }   
                  return TRUE;
              }
              
              case IDD_RESET:
              {
                  UINT i,maxch,setting;

                  olDaGetChannelType(config.hAdc,&setting);
                  if(setting=OL_CHNT_SINGLEENDED)
                    olDaGetSSCaps(config.hAdc,OLSSC_MAXSECHANS,&maxch);
                  else
                    olDaGetSSCaps(config.hAdc,OLSSC_MAXDICHANS,&maxch);

                  // set up 8 entries: channel 0-6,32;  gain 1.0
				  for (i = 0; i < 8 && i < maxch; i++) {
					olDaSetChannelListEntry( config.hAdc, i, i==7? 32 : i );
					olDaSetGainListEntry( config.hAdc, i, 1.0 );
				  }

                  for(i=0;i<maxch;i++)
                    olDaSetChannelFilter( config.hAdc, i, 0.0 );

				  //change list size
                  ListSize = min( 8, MaxCglSize );
                  SetDlgItemText( hDlg, IDD_CGLSIZE, "8" );
                  SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_RESETCONTENT, 0, 0L );
                  olDaSetChannelListSize( config.hAdc, ListSize );

                  for ( i=0;i<ListSize;i++ ) 
                  {
                     GetCGLString( &config, i, str );
                     SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
                  }
                  SelectIndex = 0;
                  SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_SETCURSEL, SelectIndex, 0L );
                  SendMessage( hDlg, WM_COMMAND, IDD_CGLSIZE, MAKELPARAM( GetDlgItem( hDlg, IDD_CGLSIZE ), EN_KILLFOCUS ) );
                  return TRUE;
              }    
              case IDD_ADDENTRY: 
              {
                  DBL gain,filter;
                  UINT channel, entry;

                  // store the new entry
                  GetDlgItemText( hDlg, IDD_CGLENTRY, str, STRLEN );
                  entry = atoi( str );
                  GetDlgItemText( hDlg, IDD_CGLGAIN, str, STRLEN );
                  gain = atof( str );
                  GetDlgItemText( hDlg, IDD_CGLCHANNEL, str, STRLEN );
                  channel = atoi( str );
                  GetDlgItemText( hDlg, IDD_CGLFILTER, str, STRLEN );
                  filter = atoi( str );
                  if( gain > 0.0 ) 
                  {
                     olDaSetGainListEntry( config.hAdc, entry, gain );
                     olDaSetChannelListEntry( config.hAdc, entry, channel );
                  }

                     olDaSetChannelFilter( config.hAdc, entry, filter );

                  // update CGL list box
                  SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_DELETESTRING, entry, 0L );
                  GetCGLString( &config, entry, str );
                  SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_INSERTSTRING, entry, ( LPARAM )( LPSTR )str );
                  entry = min( entry+1, ListSize-1 );    // select next entry
                  SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_SETCURSEL, entry, 0L );

                  //update edit box with next entry
                  wsprintf( str, "%d", entry );
                  SetDlgItemText( hDlg, IDD_CGLENTRY, str );
                  return TRUE;
              }               
                  
              case IDD_CGLLIST: 
              { 
                 WORD dlgmsg;
                 dlgmsg = HIWORD( wParam );
                 if( ( dlgmsg == LBN_DBLCLK ) || ( dlgmsg == LBN_SELCHANGE ) )
                 {
                    UINT channel;
                    DBL gain,filter;

                    SelectIndex  = ( UINT )SendMessage( ( HWND )lParam, LB_GETCURSEL, 0, 0L );

                    // always update the entry control
                    wsprintf( str, "%d", SelectIndex );
                    SetDlgItemText( hDlg, IDD_CGLENTRY, str );

                    // update the other controls only on a dbl click
                    if( HIWORD( wParam )==LBN_DBLCLK ) 
                    {
                       olDaGetChannelListEntry( config.hAdc, SelectIndex, &channel );
                       wsprintf( str, "%d", channel );
                       SendDlgItemMessage( hDlg, IDD_CGLCHANNEL, CB_SELECTSTRING, 0, ( LPARAM )( LPSTR )str );
    
                       olDaGetGainListEntry( config.hAdc, SelectIndex, &gain );
                       sprintf( str, GAIN_FORMAT, gain );
                       SendDlgItemMessage( hDlg, IDD_CGLGAIN, CB_SELECTSTRING, 0, ( LPARAM )( LPSTR )str );

                       olDaGetChannelFilter( config.hAdc, SelectIndex, &filter );
                       sprintf( str, FILTER_FORMAT, filter );
                       SendDlgItemMessage( hDlg, IDD_CGLFILTER, CB_SELECTSTRING, 0, ( LPARAM )( LPSTR )str );

                    }
                 }  
                 return TRUE;
              }   
              
              case IDD_CGLSIZE:

                  if( HIWORD( wParam ) == EN_KILLFOCUS )  
                  {
                      GetDlgItemText( hDlg, IDD_CGLSIZE, str, STRLEN );
                      i = atoi( str );
                      if( i > 0 && i != ListSize ) 
                      {
                          ListSize = min( i, MaxCglSize );
                          olDaSetChannelListSize( config.hAdc, ListSize );
                          SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_RESETCONTENT, 0, 0L );
                          for ( i=0; i<ListSize; i++ ) 
                          {
                             GetCGLString( &config, i, str );
                             SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
                          }
                          SelectIndex = 0;
                          SendDlgItemMessage( hDlg, IDD_CGLLIST, LB_SETCURSEL, SelectIndex, 0L );

                      }
                      wsprintf( str, "%d", ListSize );
                      SetDlgItemText( hDlg, IDD_CGLSIZE, str );
                  }
                  return TRUE;
              
              default:
                 break;
           }
           break;
        
       default:
           break;
           
       }
       return FALSE;               // Didn't process a message  
}

VOID 
GetCGLString( LPCONFIG lpconfig,  UINT entry,  NPSTR npstr )
/*
This function is used to generate a string that has the current
information ( channel & gain ) for the specified CGL entry of the ADC.
*/
{
    UINT channel = 0;
    DBL gain = 0,filter = 0;

    olDaGetChannelListEntry( lpconfig->hAdc, entry, &channel );
    olDaGetChannelFilter(lpconfig->hAdc,channel,&filter);
    if( olDaGetGainListEntry( lpconfig->hAdc, entry, &gain ) ) gain = 1.0;
    sprintf( npstr, "%d\t%d\t"GAIN_FORMAT"\t"FILTER_FORMAT, entry, channel, gain,filter );

}

VOID 
InitChannelComboBox( HWND hWndCombo,  HDASS hDass )
/*
This function loads the specified combo box with a string for each
channel that is available on the specified subsystem
*/
{
    UINT numchan, i;

    olDaGetSSCaps( hDass, OLSSC_NUMCHANNELS, &numchan );
    
    for ( i=0; i < numchan; i++ ) 
    {
       wsprintf( str, "%d", i );
       SendMessage( hWndCombo, CB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
    }
    SendMessage( hWndCombo, CB_SETCURSEL, 0, 0L );
}
        

BOOL CALLBACK 
AddGainList( UINT cap,  DBL gain,  DBL unused,  LPARAM lParam )
/*
This is a callback function of olDaEnumSSCaps,  it adds the
the specified gain to the specified combobox ( the handle is passed
in lParam )
*/
{
    sprintf( str, GAIN_FORMAT, gain );
    SendMessage( ( HWND )lParam, CB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
    return( TRUE );
}

BOOL CALLBACK 
AddFilterList( UINT cap,  DBL filter,  DBL unused,  LPARAM lParam )
/*
This is a callback function of olDaEnumSSCaps,  it adds the
the specified gain to the specified combobox ( the handle is passed
in lParam )
*/
{
    sprintf( str, FILTER_FORMAT, filter );
    SendMessage( ( HWND )lParam, CB_ADDSTRING, 0, ( LPARAM )( LPSTR )str );
    return( TRUE );
}


#include <stdarg.h>

BOOL CALLBACK 
SvBox( HWND hDlg,  UINT message,  WPARAM wParam,  LPARAM lParam )
/*
This is the dialog function for the single value dialog box.  It inputs
and outputs single value from / to the user specified channel at the 
user specified gain.
*/
{
    ECODE status;

    switch ( message ) 
    {
       case WM_INITDIALOG:        /* message: initialize dialog box */
          if( config.hAdc )
          {
             // set subsystem for single value operation
             olDaSetDataFlow( config.hAdc, OL_DF_SINGLEVALUE );
             status = olDaConfig( config.hAdc );
             if(  status != OLNOERROR ) 
             {
                SHOW_ERROR( hDlg, status );
                EndDialog( hDlg,  FALSE );
                return TRUE;
             }
             // init channel combo box
             InitChannelComboBox( GetDlgItem( hDlg, IDD_SVINCHANNEL ), config.hAdc );      
             // init gain combo box
             olDaEnumSSCaps( config.hAdc, 
                             OL_ENUM_GAINS, 
                             AddGainList, 
                             ( LPARAM )GetDlgItem( hDlg, IDD_SVINGAIN ) );
                             
             SendDlgItemMessage( hDlg, IDD_SVINGAIN, CB_SETCURSEL, 0, 0L );
          }
          else
          { 
             // disable GET
             EnableWindow( GetDlgItem( hDlg, IDD_SVINCHANNEL ), FALSE );
             EnableWindow( GetDlgItem( hDlg, IDD_SVINGAIN ), FALSE );
             EnableWindow( GetDlgItem( hDlg, IDD_SVGET ), FALSE );
          }
         
          if( config.hDac )
          {
             // set subsystem for single value operation
             olDaSetDataFlow( config.hDac, OL_DF_SINGLEVALUE );
             status = olDaConfig( config.hDac );
             if(  status != OLNOERROR ) 
             {
                SHOW_ERROR( hDlg, status );
                EndDialog( hDlg,  FALSE );
                return TRUE;
             }
             
             // init channel combo box
             InitChannelComboBox( GetDlgItem( hDlg, IDD_SVOUTCHANNEL ), config.hDac );     
             
             // init gain combo box
             olDaEnumSSCaps( config.hDac, 
                             OL_ENUM_GAINS, 
                             AddGainList, 
                             ( LPARAM )GetDlgItem( hDlg, IDD_SVOUTGAIN )  );
                             
             SendDlgItemMessage( hDlg, IDD_SVOUTGAIN, CB_SETCURSEL, 0, 0L );
          }
          else
          {
             // disable PUT
             EnableWindow( GetDlgItem( hDlg, IDD_SVOUTCHANNEL ), FALSE );
             EnableWindow( GetDlgItem( hDlg, IDD_SVOUTGAIN ), FALSE );
             EnableWindow( GetDlgItem( hDlg, IDD_SVPUT ), FALSE );
          }
          
          return TRUE;

    case WM_COMMAND:

        switch ( LOWORD( wParam ) ) 
        {
           case IDCANCEL: 
               EndDialog( hDlg,  TRUE );
               return TRUE;

           case IDD_SVGET: 
           {
               DBL min, max, volts, gain;
               long value;
               UINT encoding, resolution, channel;

               // get value
               GetDlgItemText( hDlg, IDD_SVINCHANNEL, str, STRLEN );
               channel = atoi( str );
               GetDlgItemText( hDlg, IDD_SVINGAIN, str, STRLEN );
               gain = atof( str );
               status = olDaGetSingleValue( config.hAdc, &value, channel, gain );
               if(  status != OLNOERROR ) {
                   SHOW_ERROR( hDlg, status );
                   return TRUE;
               }

               //convert to volts and display
               olDaGetRange( config.hAdc, &max, &min );
               olDaGetEncoding( config.hAdc, &encoding );
               olDaGetResolution( config.hAdc, &resolution );
               if( encoding != OL_ENC_BINARY ) {
                   // convert to offset binary by inverting the sign bit
                   value ^= 1L << ( resolution-1 );
                   value &= ( 1L << resolution ) - 1;     // zero upper bits
               }
               volts = ( max-min )/( 1L<<resolution ) *    value + min;
               sprintf( str, "%.3lf", volts );
               SetDlgItemText( hDlg, IDD_SVVALUE, str );
               return TRUE;
           }
           
           case IDD_SVPUT: 
           {
               DBL min, max, volts, gain;
               long value;
               UINT encoding, resolution, channel;

               olDaGetRange( config.hDac, &max, &min );
               olDaGetEncoding( config.hDac, &encoding );
               olDaGetResolution( config.hDac, &resolution );

               // get value from control and clip to current range
               GetDlgItemText( hDlg, IDD_SVVALUE, str, STRLEN );
               volts = atof( str );
               volts = min( volts, max );
               volts = max( volts, min );

               // update control with 'processed' voltage
               sprintf( str, "%.3lf", volts );
               SetDlgItemText( hDlg, IDD_SVVALUE, str );

               //convert to DAC units
               value = ( long ) ( ( 1L<<resolution )/( max-min ) * ( volts - min ) );
               value = min( ( 1L<<resolution )-1, value );

               if( encoding != OL_ENC_BINARY ) 
               {
                   // convert to 2's comp by inverting the sign bit
                   long sign = 1L << ( resolution - 1 );
                   value ^= sign;
                   if( value & sign )           //sign extend
                       value |= 0xffffffffL << resolution;
               }
               
               // put value
               GetDlgItemText( hDlg, IDD_SVOUTCHANNEL, str, STRLEN );
               channel = atoi( str );
               GetDlgItemText( hDlg, IDD_SVOUTGAIN, str, STRLEN );
               gain = atof( str );
               status = olDaPutSingleValue( config.hDac, value, channel, gain );
               if(  status != OLNOERROR )
                   SHOW_ERROR( hDlg, status );

               return TRUE;
           }
           
           default:
               break;
        }
        break;
        
     default:
        break;   
        
    }
    return FALSE;               // Didn't process a message 
}


VOID 
CloseBoard( LPCONFIG lpconfig )
/*
This function is used to close the current board.
*/                                              
{
    HBUF hbuf;

    if( lpconfig->hAdc != NULL ) 
    {
       olDaAbort( lpconfig->hAdc );
       olDaFlushBuffers( lpconfig->hAdc );
       while ( olDaGetBuffer( lpconfig->hAdc, &hbuf ) ==OLNOERROR && hbuf !=NULL )
           olDmFreeBuffer( hbuf );
       olDaReleaseDASS( lpconfig->hAdc );
       lpconfig->hAdc = NULL;
    }
    if( lpconfig->hDac != NULL ) 
    {
       olDaAbort( lpconfig->hDac );
       olDaFlushBuffers( lpconfig->hDac );
       while ( olDaGetBuffer( lpconfig->hDac, &hbuf ) ==OLNOERROR && hbuf !=NULL )
           olDmFreeBuffer( hbuf );
       olDaReleaseDASS( lpconfig->hDac );
       lpconfig->hDac = NULL;
    }
    if( lpconfig->hCT0 != NULL ) 
    {
       olDaStop( lpconfig->hCT0 );
       olDaReleaseDASS( lpconfig->hCT0 );
       lpconfig->hCT0 = NULL;
    }    

    if( lpconfig->hDev ) 
    {
       olDaTerminate( lpconfig->hDev );
       lpconfig->hDev = NULL;
    }
}


ECODE 
GenerateSignal ( LPCONFIG lpconfig )
/*
This function generates an output signal based on the parameters
in the specified config structure.  The function allocates
( or re-allocates ) an Open Layers buffer to store the results
the buffer handle is stored in the config structure.  The function
returns the error status;
*/
{
    long i;
    DBL phase,  freq,  amp,  pi2;
    DBL min, max;
    UINT res, encoding, bit;
    PSHORT wave;
    
    phase = 0;
    freq = lpconfig->dWaveFreq / lpconfig->dOutFreq;

    CHECKERROR( olDaGetRange( lpconfig->hDac, &max, &min ) );
    CHECKERROR( olDaGetResolution( lpconfig->hDac, &res ) );
    amp = lpconfig->dPeakVolts * ( 1L << res ) / ( max - min );

    //get buffer
    if( lpconfig->hDacBuf == NULL )
    {
       CHECKERROR( olDmCallocBuffer( 0, 0, lpconfig->lOutBufSize, 2, &lpconfig->hDacBuf ) );
    }
    else
    {
       CHECKERROR( olDmReCallocBuffer( 0, 0, lpconfig->lOutBufSize, 2, &lpconfig->hDacBuf ) );
    }
    
    CHECKERROR( olDmGetBufferPtr( lpconfig->hDacBuf, ( LPVOID* )&wave ) );

    //compute waveform
    if( lpconfig->uiWaveType == IDD_SINE ) 
    {
       pi2 = 3.1415927 * 2;
       freq = freq * pi2;
       for ( i=0; i < lpconfig->lOutBufSize; i++ ) 
       {
          wave[i] = ( short ) ( sin( phase ) * amp );
          phase = phase + freq;
          if( phase > pi2 )
             phase = phase - pi2;
       }
    }
    else if( lpconfig->uiWaveType == IDD_SQUARE ) 
    {
       for ( i=0; i < lpconfig->lOutBufSize; i++ ) 
       {
          wave[i] = ( short ) ( ( phase < 0 ) ? -amp : amp );
          phase = phase + freq;
          if( phase > .5 )
             phase = phase - 1;
       }
    }
    else 
    {                  //generate triangle
       for ( i = 0; i < lpconfig->lOutBufSize; i++ ) 
       {
          wave[i] = ( short ) ( 2 * phase * amp );
          phase = phase + 2 * freq;
          if( phase > .5 ) 
          {
             phase = phase - 1;
             amp = -amp;
          }         
       }
    }

    //convert from 2's comp to offset bin if required
    // ( simply invert ms and higher bits )
    CHECKERROR( olDaGetEncoding( lpconfig->hDac, &encoding ) );
    if( encoding == OL_ENC_BINARY ) 
    {
       bit =  0xffff << ( res-1 );
       for ( i = 0; i < lpconfig->lOutBufSize; i++ )
       {
          wave[i] = wave[i] ^ bit;
       }   
    }

    //set the number of valid samples & return
    return olDmSetValidSamples( lpconfig->hDacBuf, lpconfig->lOutBufSize );

}

#define BLUE 0xFF0000
#define RED 0x0FF
#define WHITE 0xFFFFFF

VOID 
PaintGraph ( HWND hwnd, LPCONFIG lpconfig )
/* 
This function paints the client window.  It draws a graph of either the DAC 
or ADC signal ( if any ) and updates the buffer done counts
*/
{
    HDC hdc;
    PAINTSTRUCT ps;
    TEXTMETRIC tm;
    int lineht;
    static float PlotWave[MAXPLOTSIZE];     
    ULONG WaveSize,tempdata;
    HANDLE hgraph;
    HDASS hdass;
    HBUF hbuf;
    int size,  TabLoc,  xTextStart;
    DBL min,  max,  freq;
    float amp,  center, endtime;
    UINT encoding,i;
    INT j,k;
    long res,resolution;
    BYTE *wave;
    UINT bufferbytes;
    ECODE status;
    long bufsize;
    BOOL usage;
    UINT source;
	char numstr[80];

    // setup for ADC or DAC plotting
    if( lpconfig->fViewDac ) 
    {
       hbuf = lpconfig->hDacBuf;
       hdass = lpconfig->hDac;
    }
    else 
    {
       if( lpconfig->fDisplay )
           hbuf = lpconfig->hAdcBuf;
       else
           hbuf = NULL;
       hdass = lpconfig->hAdc;
    }

    hdc = BeginPaint( hwnd, &ps );

    // draw the text
    if( ps.rcPaint.bottom >= lpconfig->TextRect.top ) 
    {
       GetTextMetrics( hdc, &tm );
       lineht = tm.tmHeight + tm.tmExternalLeading;
       TabLoc = 20 * tm.tmAveCharWidth * 3 / 2;
       xTextStart = 3 * tm.tmAveCharWidth;
       size = wsprintf( str, "Input Buffers done:\t%ld  Drive %04lX", lpconfig->lAdcBufsDone, driveState );
       TabbedTextOut( hdc, xTextStart, lpconfig->TextRect.top, 
                     str, size, 1, &TabLoc, xTextStart );
		if (lpconfig->dOutFreq > 100000) 
			strcpy (numstr,"---");
		else
			wsprintf (numstr,"%ld",lpconfig->lDacBufsDone );
       size = wsprintf( str, "Output Buffers done:\t%s  Timer %ld", numstr, activityTimer);
       TabbedTextOut( hdc, xTextStart, lpconfig->TextRect.top + lineht, 
                     str, size, 1, &TabLoc, xTextStart );
    }

    // draw the graph if its ready and there's room
    if( ( hbuf != NULL ) &&
        ( ( status = olDmGetValidSamples( hbuf, &bufsize ) ) == OLNOERROR ) &&
        bufsize && 
        ( lpconfig->GraphRect.bottom >= 50 ) &&
        ( ps.rcPaint.top <= lpconfig->GraphRect.bottom ) ) 
     {
    
        WaveSize = ( UINT )min( bufsize, MAXPLOTSIZE );

        // fetch board info for scaling 
        status = olDaGetRange( hdass, &max, &min );
        if( status == OLNOERROR )
            status = olDaGetResolution( hdass, &resolution );
        res = resolution;
        if( status == OLNOERROR )
            status = olDaGetClockFrequency( hdass, &freq );
        if( status == OLNOERROR )
            status = olDaGetEncoding( hdass, &encoding );
        if( status == OLNOERROR)
            status = olDaGetResolution(hdass,&bufferbytes);
          if (bufferbytes > 16)
             bufferbytes = 4;
         else
             bufferbytes = 2;
        if( status == OLNOERROR )
           status = olDmGetBufferPtr( hbuf, ( LPVOID* )&wave );
        if( status == OLNOERROR )    
        {
           res = 1L << res;
           endtime = ( float )( WaveSize / freq * 1000 );  // in milliseconds
           amp = ( float )( ( max - min ) / res );           // in volts
           center = ( float )min + amp * res/2;

           // open graph,  and plot the grid
           if( hdass == lpconfig->hAdc )
           {
              status = olDaGetClockSource( hdass, &source );
              status = olDaGetTriggeredScanUsage( hdass, &usage );
              if( ( source == OL_CLK_EXTERNAL )||( usage == TRUE ) )
              {
                 hgraph = GraphOpen( hdc,  &lpconfig->GraphRect,  1,  "Data",  "Samples", 
                                  "Volts",  BLUE,  BLUE,  BLUE,  0,  bufsize,  bufsize / 10,  0, 
                                  min,  max,  ( max - min ) / 5,  0 );
              }                    
              else
              {
                 hgraph = GraphOpen( hdc,  &lpconfig->GraphRect,  1,  "Data",  "Milliseconds", 
                                  "Volts",  BLUE,  BLUE,  BLUE,  0,  endtime,  endtime / 10,  0, 
                                  min,  max,  ( max - min ) / 5,  0 );
              }                   
           }
           else
           {
                hgraph = GraphOpen( hdc,  &lpconfig->GraphRect,  1,  "Data",  "Milliseconds", 
                                    "Volts",  BLUE,  BLUE,  BLUE,  0,  endtime,  endtime / 10,  0, 
                                    min,  max,  ( max - min ) / 5,  0 );
           }      
           
           //erase old plot if any
           if( lpconfig->uiOldWaveSize )
           {
              GraphFltSignal( hgraph,  ( float ) 0,  1000 / ( float )freq,  PlotWave, 
                              lpconfig->uiOldWaveSize,  WHITE,  0 );
           }                   
            
           //   scale the new data
           if( encoding == OL_ENC_BINARY ) 
           {
              //perform scaling for offset binary encoding
              j=0;
              for ( i=0; i<WaveSize*bufferbytes;i=i+bufferbytes)    // Do Byte operations - not as efficient - but works for multiple buffer widths
              {
                tempdata = 0;
                for (k=bufferbytes-1;k>0;k--)       // Properly assemble the Data
                {
                tempdata = tempdata+(BYTE)wave[i+k];
                tempdata<<=8;
                }
                tempdata = tempdata+(BYTE)wave[i];

                 if( tempdata < 0 )
                    PlotWave[j] = ( tempdata + res ) * amp + ( float )min;
                 else
                    PlotWave[j] = (tempdata) * amp + ( float )min;
                j++;
              }
           }
           else 
           {
              //perform scaling for 2's comp encoding
              j=0;
              for ( i=0; i<WaveSize*bufferbytes;i=i+bufferbytes)       // Do Byte operations - not as efficient - but works for multiple buffer widths
              {
               tempdata = 0;
                for (k=bufferbytes-1;k>0;k--)       // Properly assemble the Data
                {
                tempdata = tempdata+(BYTE)wave[i+k];
                tempdata<<=8;
                }
                tempdata = tempdata+(BYTE)wave[i];    

               tempdata ^= 1L << (resolution-1);
               tempdata &= (1L << resolution) - 1;     /* zero upper bits */
 
            PlotWave[j]= ((float)max-(float)min)/(1L<<resolution)*tempdata + (float)min;


                 j++;
              }   
           }
        
           //plot the new data
           GraphFltSignal( hgraph,  ( float )0,  1000 / ( float )freq,  PlotWave,  WaveSize,  RED,  0 );
           GraphClose( hgraph );
           lpconfig->uiOldWaveSize = WaveSize;

        }
    }

    EndPaint( hwnd, &ps );
}

ECODE 
StartAcq( LPCONFIG lpconfig )
/*
This function starts data acquisition using setup parameters in the 
specified config structure.  It also updates the control / status
fields in the config struct ( these include buffer done counts and the
FViewDac flag. )  The function returns the appropriate result code.
*/
{
    ECODE status;
    HBUF hbuf;
    UINT i;
    UINT bufferbytes;
	HANDLE fh;

	if (!getDriveReady()) {
		return OLNOTRUNNING;
	}
	currentTimeout = -1;
	stopCounter = -1;
	forward = getDriveBot();

    if( !lpconfig->fDisableIn && lpconfig->fFileWrite ) 
	{
		for (;;)
		{
			fh = CreateFile(
				adcfile,
				GENERIC_WRITE,
				0,
				NULL,
				CREATE_NEW,
				FILE_ATTRIBUTE_ARCHIVE,
				NULL);

			if( fh == INVALID_HANDLE_VALUE ) {
				if (AdcfileUpdate())
				{
					continue;
				}
				return -1;
			}
			CloseHandle(fh);
			break;
		}
	}

	// setup the DAC
    if( !lpconfig->fDisableOut ) 
    {
        CHECKERROR( olDaSetDataFlow( lpconfig->hDac, OL_DF_CONTINUOUS ) );
        
        // Set the DAC for wrap mode to generate a repetitive waveform.
        // We could use OL_WRP_SINGLE ( which takes advantage of auto-init dma
        // for gap free transfers ),  but buffer done messages are not generated.
        // OL_WRP_MULTIPLE mode generates buffer done messages,  which we use to
        // increment the DAC buffer done count.
        //
		if (lpconfig->dOutFreq > 100000) 
	        CHECKERROR( olDaSetWrapMode( lpconfig->hDac, OL_WRP_SINGLE ) );
		else
	        CHECKERROR( olDaSetWrapMode( lpconfig->hDac, OL_WRP_MULTIPLE ) );
        CHECKERROR( olDaSetClockFrequency( lpconfig->hDac, lpconfig->dOutFreq ) );
        CHECKERROR( olDaSetDmaUsage( lpconfig->hDac, lpconfig->uiOutDma ) );
        CHECKERROR( olDaConfig( lpconfig->hDac ) );
    }

    // set up the ADC
    if( !lpconfig->fDisableIn ) 
    {
        CHECKERROR( olDaSetDataFlow( lpconfig->hAdc, OL_DF_CONTINUOUS ) );
        CHECKERROR( olDaSetWrapMode( lpconfig->hAdc, OL_WRP_NONE ) );
        CHECKERROR( olDaSetClockFrequency( lpconfig->hAdc, lpconfig->dInFreq ) );
        CHECKERROR( olDaSetDmaUsage( lpconfig->hAdc, lpconfig->uiInDma ) );
        CHECKERROR( olDaSetTriggeredScanUsage( lpconfig->hAdc, lpconfig->fScanEnable ) );
        
        if( lpconfig->fScanEnable ) 
        {
           CHECKERROR( olDaSetRetriggerFrequency( lpconfig->hAdc, lpconfig->dScanFreq ) );
        }
        CHECKERROR( olDaConfig( lpconfig->hAdc ) );
    }

    // allocate and fill the output buffer
    if( !lpconfig->fDisableOut ) 
    {
       status = GenerateSignal( lpconfig );
       if( status != OLNOERROR )
           return status;
           
       CHECKERROR( olDaPutBuffer( lpconfig->hDac, lpconfig->hDacBuf ) );
       lpconfig->hDacBuf = NULL;  // the dac buffer is now owned by the device
    }

    // allocate input buffers
    if( !lpconfig->fDisableIn ) 
    {
        UINT numbuf;
        if( lpconfig->hAdcBuf != NULL ) 
        {  
           // remove old ADC display buf ( if any )
           olDmFreeBuffer( lpconfig->hAdcBuf );
           lpconfig->hAdcBuf = NULL;
        }
        numbuf = lpconfig->uiNumBuf;
        
        // in continuous mode,  allocate 1 extra buffer to hold display data
        if( lpconfig->uiDataFlow == IDD_CONTINUOUS )
            numbuf++;
        
        CHECKERROR(olDaGetResolution(lpconfig->hAdc,&bufferbytes));
        if (bufferbytes > 16)
            bufferbytes = 4;
        else
            bufferbytes = 2;

        for ( i=0; i < numbuf; i++ ) 
        {
           CHECKERROR( olDmCallocBuffer( GMEM_FIXED, 0, lpconfig->ulBufSize, bufferbytes, &hbuf ) );
           CHECKERROR( olDaPutBuffer( lpconfig->hAdc, hbuf ) );
        }

        status = olDmCallocBuffer( 0, 0, lpconfig->ulBufSize/2, bufferbytes, &( lpconfig->hFlushBuf ) );

        if( status != OLNOERROR ) 
        {
           StopAcq( lpconfig );
           return status;
        }
    }

    //start-em
    lpconfig->fViewDac = FALSE;
    lpconfig->lAdcBufsDone = 0;
    lpconfig->lDacBufsDone = 0;

    if( !lpconfig->fDisableOut )
        status = olDaStart( lpconfig->hDac );

    if( !lpconfig->fDisableIn && status == OLNOERROR )
        status = olDaStart( lpconfig->hAdc );

	if( status != OLNOERROR ) {
        StopAcq( lpconfig );
		return status;
	}

	if (forward) {
		setDriveForward();
	}
	else
	{
		setDriveReverse();
	}

    return status;
}

VOID 
StopAcq( LPCONFIG lpconfig )
/*
This function stops data acquisition and frees all
buffers "owned" by the board.
*/
{
    HBUF hbuf;
    char msg[80];

	setDriveStop();

    if( !lpconfig->fDisableIn ) 
    {
       olDaAbort( lpconfig->hAdc );
       olDaFlushBuffers( lpconfig->hAdc );
       while ( olDaGetBuffer( lpconfig->hAdc, &hbuf ) == OLNOERROR && hbuf != NULL )
           olDmFreeBuffer( hbuf );
       olDmFreeBuffer( lpconfig->hFlushBuf );
           
       if( config.fFileWrite ) 
       { 
          lstrcpy( msg, "Data written to " );
          lstrcat( msg, adcfile );
          MessageBox( NULL, msg, "Acquisition Stopped",  MB_ICONEXCLAMATION | MB_OK ); 
		  AdcfileUpdate();
       }
        
    }
    if( !lpconfig->fDisableOut ) 
    {
       olDaAbort( lpconfig->hDac );
       olDaFlushBuffers( lpconfig->hDac );
       while ( olDaGetBuffer( lpconfig->hDac, &hbuf ) == OLNOERROR && hbuf != NULL )
           olDmFreeBuffer( hbuf );
    }
}

VOID
initTimers()
{
	currentTimeout = normalTimeout;
	activityTimer = currentTimeout;
	currentPolarity = 0;
	driveState = 0;
}

VOID
initOffset( unsigned short _huge *dp)
{
	int i, j;
	for (j = 0; j < 7; j++) {
		offset[j] = 0;
	}
	for (i = 0; i < 100; i++) {
		for (j = 0; j < 7; j++) {
			offset[j] += *dp++ - 0x800;
		}
		dp++;
	}
	for (j = 0; j < 7; j++) {
		offset[j] = offset[j]/100;
	}
}

BOOL
processData( unsigned short _huge *dp, long dn)
{
	unsigned short v;
	unsigned long ct;
	unsigned long at;
	unsigned long cp;
	unsigned long i;

	if (currentTimeout == -1) {
		initTimers();
		initOffset(dp);
	}
	ct = currentTimeout;
	at = activityTimer;
	cp = currentPolarity;
	for (; at > 0 && dn > 0; ) {
		if (forward) {
			--at;
			if (currentTimeout == 0) {
				ct = at;
			}
			for (i = 0; i < 7; i++) {
				v = *dp++ - offset[i];
				if (v > 0x0B00) {
					cp |= 1 << i;
					at = ct;
				} else if (v < 0x0500) {
					cp &= ~(1 <<i);
					at = ct;
				}
			}
			v = *dp++;
			if ((v & (DRIVE_EOT << 8)) == 0) {
				currentTimeout = 0;
				if (at > eotTimeout) {
					at = currentTimeout;
				}
			}
		}
		else
		{
			dp += 7;
			v = *dp++;
		}
		if ((v & (DRIVE_BOT << 8)) == 0 &&
			(driveState & (DRIVE_BOT << 8)) != 0 ||
			(v & (DRIVE_ONLINE << 8)) != 0) {
			at = 0;
			break;
		}
		dn -= 16;
	}
	driveState = v;
	activityTimer = at;
	currentPolarity = cp;
	return at == 0;
}

BOOL 
WriteBufferToFile( HWND hWnd, HBUF hBuf,  LPSTR lpstr,  UINT strlen )
/*
This function writes the specified buffer to the adc output file:
adcfile.  The function returns TRUE if successful.  Otherwise, 
it returns FALSE and writes an error message to the specified
string ( strlen is the max length of the input string )
*/

{
    HFILE hFile;
    BYTE _huge *dataptr = NULL;
    long bufsize=0L;
    UINT size=0L;
    ECODE status=OLNOERROR;
    BOOL rval=FALSE;
	BOOL stop;

    status = olDmGetValidSamples( hBuf, &bufsize );
    if( status == OLNOERROR )
        status = olDmGetDataWidth( hBuf, &size );
    if( status == OLNOERROR )
        status = olDmGetBufferPtr( hBuf, &( LPVOID )dataptr );
    if( status != OLNOERROR ) {
        olDaGetErrorString( status, lpstr, strlen );
       return FALSE;
    }

	hFile = _lopen( adcfile, OF_WRITE | OF_SHARE_EXCLUSIVE );
    if( hFile == HFILE_ERROR ) {
       lstrcpyn( lpstr, "Can't create output file", strlen );  
       return FALSE;
    }

    //un-comment next line to fill disk
    _llseek( hFile, 0, 2 );

    // write the data
    rval = TRUE;
    bufsize *= size;

	stop = processData((unsigned short *)dataptr, bufsize);

    while ( size=( UINT )min( 0xfffe, bufsize ) ) 
    {
       size = _lwrite( hFile, dataptr, size );
       if( size == HFILE_ERROR || size == 0 ) 
       {
          rval = FALSE;
          lstrcpyn( lpstr, "Can't write to output file", strlen );
          break;
       }
       dataptr += size;
       bufsize -= size;
    }

    _lclose( hFile );
	if (stop) {
		setDriveStop();
		if (stopCounter == -1) {
			stopCounter = 5;
		}
	}
	if (stopCounter > 0) {
		stopCounter--;
		if (stopCounter == 0) {
           SendMessage( hWnd, WM_COMMAND, IDM_STOP, 0L );
		}
	}
    return rval;
}

BOOL
AdcfileUpdate()
/*
 * Update the filename adcfile to another filename
 * if there is a trailing numeric part of it
 */
{
	char *p;

	for (p = adcfile; *p && *p != '.'; p++);
	for (;;) {
		--p;
		if (p < adcfile) {
			return FALSE;
		}
		if (*p >= '0' && *p < '9') {
			(*p)++;
			return TRUE;
		}
		else if (*p == '9') {
			*p = '0';
		}
		else
		{
			return FALSE;
		}
	}
}

#define GEN_RATE "Generate Rate"
#define ONE_SHOT "One Shot"
#define ONE_SHOT_RETRIG "Re-Triggerable One Shot"
#define COUNT_EVENTS    "Count Events"
#define RUNNING_TITLE "Counter / Timer 0 - Running"
#define STOPPED_TITLE "Counter / Timer 0 - Stopped"

BOOL CALLBACK 
CTBox( HWND hDlg,  UINT message,  WPARAM wParam,  LPARAM lParam )
/*
This is the dialog function for the counter/timmer dialog box.  It
configures and runs CT 0 based on user input.
*/
{
    HWND hItem;
    ECODE status;

    switch ( message ) 
    {
       case WM_INITDIALOG: 
       {      
           DBL ClkFreq;
           DBL width;
           UINT sup, item;

           if( config.hCT0 == NULL ) 
           {
              config.fCTRunning = FALSE;
              config.uiCTModeItem = 0;
              status = olDaGetDASS( config.hDev, OLSS_CT, 0, &config.hCT0 );
              if( status != OLNOERROR ) 
              {
                 SHOW_ERROR( hDlg, status );
                 EndDialog( hDlg,  FALSE );
                 return TRUE;
              }
           }

           // Fill CT mode Combo Box for each supported mode
           // and,  load item data with the corresponding CT mode
           olDaGetSSCaps( config.hCT0, OLSSC_SUP_CTMODE_RATE, &sup );
           if( sup ) 
           {
              item=( UINT )SendDlgItemMessage( hDlg, IDD_CTMODE, CB_ADDSTRING, 0, ( LPARAM ) ( LPSTR ) GEN_RATE );
              SendDlgItemMessage( hDlg, IDD_CTMODE, CB_SETITEMDATA, item, ( LPARAM )OL_CTMODE_RATE );
           }
           olDaGetSSCaps( config.hCT0, OLSSC_SUP_CTMODE_ONESHOT, &sup );
           if( sup ) 
           {
              item=( UINT )SendDlgItemMessage( hDlg, IDD_CTMODE, CB_ADDSTRING, 0, ( LPARAM ) ( LPSTR ) ONE_SHOT );
              SendDlgItemMessage( hDlg, IDD_CTMODE, CB_SETITEMDATA, item, ( LPARAM )OL_CTMODE_ONESHOT );
           }
           olDaGetSSCaps( config.hCT0, OLSSC_SUP_CTMODE_ONESHOT_RPT, &sup );
           if( sup ) 
           {
              item=( UINT )SendDlgItemMessage( hDlg, IDD_CTMODE, CB_ADDSTRING, 0, ( LPARAM ) ( LPSTR ) ONE_SHOT_RETRIG );
              SendDlgItemMessage( hDlg, IDD_CTMODE, CB_SETITEMDATA, item, ( LPARAM )OL_CTMODE_ONESHOT_RPT );
           }
           olDaGetSSCaps( config.hCT0, OLSSC_SUP_CTMODE_COUNT, &sup );
           if( sup ) 
           {
              item=( UINT )SendDlgItemMessage( hDlg, IDD_CTMODE, CB_ADDSTRING, 0, ( LPARAM ) ( LPSTR ) COUNT_EVENTS );
              SendDlgItemMessage( hDlg, IDD_CTMODE, CB_SETITEMDATA, item, ( LPARAM )OL_CTMODE_COUNT );
           }
           SendDlgItemMessage( hDlg, IDD_CTMODE, CB_SETCURSEL,  config.uiCTModeItem,  0L );


           // init other controls
           olDaGetClockFrequency( config.hCT0, &ClkFreq );
           sprintf( str, FREQ_FORMAT, ClkFreq );        
           SetDlgItemText( hDlg, IDD_CTFREQ, str );
           olDaGetPulseWidth( config.hCT0, &width );
           sprintf( str, WIDTH_FORMAT, width );     
           SetDlgItemText( hDlg, IDD_CTWIDTH, str );

           // set hardware to new mode & Gray out invalid options
           SetCTMode( hDlg,  config.uiCTModeItem );

           if( config.fCTRunning ) 
           {
              // disable the following controls when running
              EnableWindow( GetDlgItem( hDlg, IDD_CTSTART ), FALSE );
              EnableWindow( GetDlgItem( hDlg, IDD_CTFREQ ), FALSE ); 
              EnableWindow( GetDlgItem( hDlg, IDD_CTWIDTH ), FALSE );
              SetWindowText( hDlg, RUNNING_TITLE );
           }
           else 
           {
              EnableWindow( GetDlgItem( hDlg, IDD_CTSTOP ), FALSE );
              SetWindowText( hDlg, STOPPED_TITLE );
           }
           return TRUE;
       }
    
       case WM_COMMAND:

          switch ( LOWORD( wParam ) ) 
          {

             case IDD_CTSTART: 
             {
                 ULNG freq;
                 DBL width;
                 UINT CTmode;

                 // fetch CT mode
                 olDaGetCTMode( config.hCT0, &CTmode );

                 olDaSetTrigger( config.hCT0, OL_TRG_SOFT );
                 olDaSetGateType( config.hCT0, OL_GATE_NONE );
                 if( CTmode == OL_CTMODE_COUNT ) 
                 {
                    olDaSetClockSource( config.hCT0, OL_CLK_EXTERNAL );
                 }
                 else 
                 {
                    if( CTmode == OL_CTMODE_ONESHOT_RPT ) 
                    {
                       olDaSetTrigger( config.hCT0, OL_TRG_EXTERN );
                       olDaSetGateType( config.hCT0, OL_GATE_HIGH_EDGE );
                    }
                    olDaSetClockSource( config.hCT0, OL_CLK_INTERNAL );
                    GetDlgItemText( hDlg, IDD_CTFREQ, str, STRLEN );
                    freq = atol( str );
                    olDaSetClockFrequency( config.hCT0, freq );
                    GetDlgItemText( hDlg, IDD_CTWIDTH, str, STRLEN );
                    width = atof( str );
                    olDaSetPulseWidth( config.hCT0, width );
                    hItem = GetDlgItem( hDlg, IDD_CTFREQ );
                 }

                 // go for it    
                 status = olDaConfig( config.hCT0 );
                 
                 if( status == OLNOERROR )
                     status = olDaStart( config.hCT0 );
                 
                 if( status == OLNOERROR ) 
                 {
                    // start ok,  enable stop button,  gray start button
                    config.fCTRunning = TRUE;
                    SetWindowText( hDlg, RUNNING_TITLE );
                    EnableWindow( GetDlgItem( hDlg, IDD_CTSTOP ), TRUE ); 
                    EnableWindow( GetDlgItem( hDlg, IDD_CTSTART ), FALSE );
                    // gray Frequency and width controls
                    EnableWindow( GetDlgItem( hDlg, IDD_CTFREQ ), FALSE ); 
                    EnableWindow( GetDlgItem( hDlg, IDD_CTWIDTH ), FALSE );

                 }   
                 else
                     SHOW_ERROR( hDlg, status );
                     
                 return TRUE;
             }
             
             case IDD_CTMODE: 
             {
                 // trap change in CT function - to gray out invalid options   
                 
                  if( HIWORD( wParam )  == CBN_SELCHANGE ) 
                  {
                      // stop current operation ( in any )

                     SendMessage( hDlg, WM_COMMAND, IDD_CTSTOP, 0L );
                     
                     // fetch new CT mode
                     config.uiCTModeItem = ( UINT )SendMessage( ( HWND )lParam,  CB_GETCURSEL,  0,  0L );

                     // Set new CT mode & Gray invalid options
                     SetCTMode( hDlg,  config.uiCTModeItem );

                     return TRUE;
                 }
                 break;
             }

             case IDD_CTSTOP: 
             {
                 UINT CTMode;
                 olDaStop( config.hCT0 );
                 config.fCTRunning = FALSE;
                 SetWindowText( hDlg, STOPPED_TITLE );
                 EnableWindow( GetDlgItem( hDlg, IDD_CTSTOP ), FALSE );    
                 EnableWindow( GetDlgItem( hDlg, IDD_CTSTART ), TRUE );

                 // enable Frequency and width controls if not count mode
                 olDaGetCTMode( config.hCT0, &CTMode );
                 if( CTMode != OL_CTMODE_COUNT ) 
                 {
                    EnableWindow( GetDlgItem( hDlg, IDD_CTFREQ ), TRUE ); 
                    EnableWindow( GetDlgItem( hDlg, IDD_CTWIDTH ), TRUE );
                 }
                 return TRUE;
             }

             case IDD_CTREAD: 
             {
                 ULNG events;
                 olDaReadEvents( config.hCT0, &events );
                 wsprintf( str, "%ld", events );
                 SetDlgItemText( hDlg, IDD_EVENT, str );
                 return TRUE;
             }

             case IDCANCEL:
                 EndDialog( hDlg,  TRUE );
                 return TRUE;

             case IDD_CTFREQ:
                 // when freq edit box is changed,  set and read back freq
                 // to display the exact freq that will be used

                 if( HIWORD( wParam )==EN_KILLFOCUS ) 
                 {  
                     DBL freq;
                     GetWindowText( ( HWND )lParam, str, STRLEN );
                     freq = atof( str );
                     status = olDaSetClockFrequency( config.hCT0, freq );
                     if( status == OLNOERROR )
                         status = olDaConfig( config.hCT0 );
                     if( status != OLNOERROR )
                         SHOW_ERROR( hDlg, status );
                     olDaGetClockFrequency( config.hCT0, &freq );
                     sprintf( str, FREQ_FORMAT, freq );
                     SetWindowText( ( HWND )lParam, str );
                     return TRUE;
                 }

             case IDD_CTWIDTH:

                 // when freq edit box is changed,  set and read back freq
                 // to display the exact freq that will be used

                 if( HIWORD( wParam )==EN_KILLFOCUS ) 
                 {  
                     DBL width;
                     GetWindowText( ( HWND )lParam, str, STRLEN );
                     width = atof( str );
                     status = olDaSetPulseWidth( config.hCT0, width );
                     if( status == OLNOERROR )
                         status = olDaConfig( config.hCT0 );
                     if( status != OLNOERROR )
                         SHOW_ERROR( hDlg, status );
                     olDaGetPulseWidth( config.hCT0, &width );
                     sprintf( str, WIDTH_FORMAT, width );
                     SetWindowText( ( HWND )lParam, str );
                     return TRUE;
                 }
             
             default:
                break;
          }
          break;
           
       default:
          break;
    }
    return FALSE;               // Didn't process a message    
}

VOID 
SetCTMode( HWND hDlg,  UINT index )
/*
This function sets the CT mode based on the "index" parameter.  "index"
is an index into the IDD_CTMODE combo box.  This function also grays the
appropriate CT dialog options bassed on the current CT mode.
*/
{
    UINT CTmode;
    BOOL CountMode;
    HWND hItem;

    //fetch mode
    CTmode = ( UINT )SendDlgItemMessage( hDlg,  IDD_CTMODE,  CB_GETITEMDATA,  index,  0L );
    CountMode = ( CTmode == OL_CTMODE_COUNT );

    //gray/enable items used when mode != count
    hItem = GetDlgItem( hDlg, IDD_CTFREQ );
    EnableWindow( hItem, !CountMode );  
    hItem = GetDlgItem( hDlg, IDD_FREQSTR );
    EnableWindow( hItem, !CountMode );  
    hItem = GetDlgItem( hDlg, IDD_CTWIDTH );
    EnableWindow( hItem, !CountMode );  
    hItem = GetDlgItem( hDlg, IDD_WIDTHSTR );
    EnableWindow( hItem, !CountMode );
      
    // gray/enable items used when mode == count
    hItem = GetDlgItem( hDlg, IDD_EVENTSTR );
    EnableWindow( hItem, CountMode );   
    hItem = GetDlgItem( hDlg, IDD_EVENT );
    EnableWindow( hItem, CountMode );   
    hItem = GetDlgItem( hDlg, IDD_CTREAD );
    EnableWindow( hItem, CountMode );   

    //set hardware to the new mode
    olDaSetCTMode( config.hCT0,  CTmode );
    olDaConfig( config.hCT0 );

}


/*
This is the dialog function for the Digital I/O dialog box.  It
configures and runs the digital I/O subsystems based on user input.
*/
BOOL CALLBACK 
DigIOBox( HWND hDlg,  UINT message,  WPARAM wParam,  LPARAM lParam )
{                                   
    long value;
    UINT element,  resolution;
    UINT channel = 0;
    DBL gain = 1.0;   
    
    HDASS hDIN,  hDOUT;
    char msg[80];
    
    
    ECODE status;
                      
    static OLSS IOtype;    
            

    switch ( message ) 
    {
       case WM_INITDIALOG:       //  initialize dialog box 
          if( config.hDigIn0 == NULL )
          { 
             olDaGetDASS( config.hDev, OLSS_DIN,  0,  &config.hDigIn0 );
          }
          
          if( config.hDigIn1 == NULL )
          { 
             olDaGetDASS( config.hDev, OLSS_DIN,  1,  &config.hDigIn1 );
          }
          
          if( config.hDigIn2 == NULL )
          { 
             olDaGetDASS( config.hDev, OLSS_DIN,  2,  &config.hDigIn2 );
          }
          
          if( config.hDigIn3 == NULL )
          { 
             olDaGetDASS( config.hDev, OLSS_DIN,  3,  &config.hDigIn3 );
          }   
       
          
          if( config.hDigOut0 == NULL )
          { 
             olDaGetDASS( config.hDev, OLSS_DOUT,  0,  &config.hDigOut0 );
          }
          
          if( config.hDigOut1 == NULL )
          { 
              olDaGetDASS( config.hDev, OLSS_DOUT,  1,  &config.hDigOut1 );
          }
          
          if( config.hDigOut2 == NULL )
          { 
              olDaGetDASS( config.hDev, OLSS_DOUT,  2,  &config.hDigOut2 );
          }
          
          if( config.hDigOut3 == NULL )
          { 
              olDaGetDASS( config.hDev, OLSS_DOUT,  3,  &config.hDigOut3 );
          }
                      
       
          //set defaults
          
          //set value to 0 
          wsprintf( str,  "%u", 0 );     
          SetDlgItemText( hDlg, IDC_VALUE, str );                     
          
          // set resolution
          wsprintf( str,  "%u", 8 );     
          SetDlgItemText( hDlg, IDC_RESOLUTION, str );                        
          
          //set subsystem element to 0;      
          wsprintf( str,  "%u", 0 );     
          SetDlgItemText( hDlg, IDC_ELEMENT_IN, str );
          
          //set subsystem element to 1;      
          wsprintf( str,  "%u", 1 );     
          SetDlgItemText( hDlg, IDC_ELEMENT_OUT, str );
       
          return TRUE;
    
       
       case WM_COMMAND:                 
          switch ( LOWORD( wParam ) ) 
          {
             case IDC_GET:     //  Get Button pressed.
                 // Get Sub system element # from dialog box
                 GetDlgItemText( hDlg, IDC_ELEMENT_IN, str, STRLEN );
                 element = ( UINT ) atol( str );            
                 
                 switch ( element )
                 {
                    case 0:
                       hDIN = config.hDigIn0;
                       if( hDIN == NULL )
                       {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                       }                                           
                       break;
                    
                    case 1:                   
                       hDIN = config.hDigIn1;                  
                       if( hDIN == NULL )
                       {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                       }                                           
                       break;                    
                        
                    
                    case 2:                   
                       hDIN = config.hDigIn2;                      
                       if( hDIN == NULL )
                       {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                       }                                           
                       break;
                    
                    case 3:
                       hDIN = config.hDigIn3;                      
                       if( hDIN == NULL )
                       {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                       }                                           
                       break;                                                              
                    
                    default:
                       wsprintf( msg,  "Element # %d is not supported.\nCExample only supports <= 3.",  element );
                       MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                       return TRUE;                    
                        
                 } // end switch( element )
                 
                 // set subsystem for single value operation
                 //      
                 status = olDaSetDataFlow( hDIN, OL_DF_SINGLEVALUE );
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );
                 }
                 
                 // set the resolution
                 GetDlgItemText( hDlg, IDC_RESOLUTION, str, STRLEN );
                 resolution = ( UINT ) atol( str );
                 
                 status = olDaSetResolution ( hDIN,  resolution );
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );               
                 }
                 
                 status = olDaConfig( hDIN );   
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );
                 }
                 
                 status = olDaGetSingleValue( hDIN, &value, channel, gain );
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );
                 }
                 
                 //Update value in dialog box
                 wsprintf( str,  "%u", value );     
                 SetDlgItemText( hDlg, IDC_VALUE, str );             
                 
                 status = olDaGetResolution ( hDIN,  &resolution ); 
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );                   
                 }
                     
                 // Set the actual achieved resoulution
                 wsprintf( str,  "%u", resolution );     
                 SetDlgItemText( hDlg, IDC_RESOLUTION, str );                                    
                                
                 return TRUE;
             
             
             case IDC_PUT:           
                 // Get Value from dialog box
                 GetDlgItemText( hDlg, IDC_VALUE, str, STRLEN );
                 value = atol( str );
                 
                 // Get Sub system element # from dialog box
                 GetDlgItemText( hDlg, IDC_ELEMENT_OUT, str, STRLEN );
                 element = ( UINT ) atol( str );  
                 
                 
                 switch ( element )
                 {
                    case 0:
                        hDOUT = config.hDigOut0;
                        if( hDOUT == NULL )
                        {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                        }                                           
                        break;
                    
                    case 1:                   
                        hDOUT = config.hDigOut1;                    
                        if( hDOUT == NULL )
                        {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                        }                                           
                        break;                    
                        
                    
                    case 2:                   
                        hDOUT = config.hDigOut2;                        
                        if( hDOUT == NULL )
                        {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                        }                                           
                        break;
                    
                    case 3:
                        hDOUT = config.hDigOut3;                        
                        if( hDOUT == NULL )
                        {       
                           wsprintf( msg,  "Element # %d is not supported",  element );
                           MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                           return TRUE;
                        }                                           
                        break;                                                              
                    
                    default:
                        wsprintf( msg,  "Element # %d is not supported.\nCExample only supports <= 3.",  element );
                        MessageBox ( hDlg,  msg,  NULL,  MB_ICONEXCLAMATION | MB_OK );                   
                        return TRUE;                    
                         
                 
                 } // end switch( element )                                                        
                 
                                                                                            
                 // set subsystem for single value operation 
                 status = olDaSetDataFlow( hDOUT, OL_DF_SINGLEVALUE );   
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );                                           
                 }
                     
                 // set the resolution
                 GetDlgItemText( hDlg, IDC_RESOLUTION, str, STRLEN );
                 resolution = ( UINT ) atol( str );
                 
                 status = olDaSetResolution ( hDOUT,  resolution );             
                 if( status != OLNOERROR )                       
                 {               
                     SHOW_ERROR( hDlg,  status );                                                       
                 }
                 
                 status = olDaConfig( hDOUT );                                 
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );
                 }
                 
                 status = olDaPutSingleValue( hDOUT, value, channel, gain );          
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );               
                 }
                     
                 // get resoulution
                 status = olDaGetResolution ( hDOUT,  &resolution ); 
                 if( status != OLNOERROR )
                 {
                     SHOW_ERROR( hDlg,  status );                   
                 }
                     
                 // Set the actual achieved resoulution
                 wsprintf( str,  "%u", resolution );     
                 SetDlgItemText( hDlg, IDC_RESOLUTION, str );                    
                 return TRUE;          
             
             case IDCANCEL:  
                 EndDialog( hDlg,  TRUE );
                 return TRUE;
                 
    
             default:
                break;
          }
          break;
          
       default:
          break;
    }

    return FALSE;               // Didn't process a message    

}  //  end DigIOBox(  )



VOID initDriveControl()
{
    HDASS hDOUT;
    HDASS hDIN;
    ECODE status;

    olDaGetDASS( config.hDev, OLSS_DOUT,  0,  &config.hDigOut0 );
    hDOUT = config.hDigOut0;
	if (hDOUT == NULL) {
		return;
	}

	// set subsystem for single value operation 
    status = olDaSetDataFlow( hDOUT, OL_DF_SINGLEVALUE );   
    if( status != OLNOERROR )
    {
        return;                                           
    }
        
    // set the resolution
    status = olDaSetResolution ( hDOUT,  8 );             
    if( status != OLNOERROR )                       
    {               
        return;                                           
    }
    
    status = olDaConfig( hDOUT );                                 
    if( status != OLNOERROR )
    {
        return;                                           
    }
    
    olDaGetDASS( config.hDev, OLSS_DIN,  1,  &config.hDigIn1 );
    hDIN = config.hDigIn1;                  

    // set subsystem for single value operation
    //      
    status = olDaSetDataFlow( hDIN, OL_DF_SINGLEVALUE );
    if( status != OLNOERROR )
    {
        return;
    }
    
    // set the resolution
    status = olDaSetResolution ( hDIN,  8 );
    if( status != OLNOERROR )
    {
        return;               
    }
    
    status = olDaConfig( hDIN );   
    if( status != OLNOERROR )
    {
        return;
    }
}

VOID setDriveCommand(long value)
{
    HDASS hDOUT;
    ECODE status;

    if( config.hDigOut0 == NULL )
    { 
		initDriveControl();
    }
    hDOUT = config.hDigOut0;
	if (hDOUT == NULL) {
		return;
	}
    
    status = olDaPutSingleValue( hDOUT, value, 0, 1.0 );
    if( status != OLNOERROR )
    {
        return;                                           
    }
}

long getDriveStatus()
{
    long value;
    HDASS hDIN;
    ECODE status;

    if( config.hDigIn1 == NULL )
    { 
		setDriveCommand(0xFF);
    }
    hDIN = config.hDigIn1;                  

    status = olDaGetSingleValue( hDIN, &value, 0, 1.0 );
    if( status != OLNOERROR )
    {
        return -1;
    }
	return value;
}

BOOL getDriveReady()
{
	return (getDriveStatus() & DRIVE_READY) == 0;
}

BOOL getDriveBot()
{
	return (getDriveStatus() & DRIVE_BOT) == 0;
}

VOID setDriveForward()
{
	setDriveCommand(DRIVE_FWD);
}

VOID setDriveReverse()
{
	setDriveCommand(DRIVE_REV);
}

VOID setDriveStop()
{
	setDriveCommand(0xFF);
}

