#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>

#define MAXREC			(32768*6 + 1000)

#define FILEMARK		0x0F
#define RECORDMARK			0x80

#define COUNTS_PER_INCH	30000
#define INVISIBLE_COUNT	5
#define GAP_CORRECTION	1

#define LRC_GAP_CHAR	4
#define RECORD_GAP_COUNT (3*COUNTS_PER_INCH/4)

int density[] = { 200, 556, 800 };

unsigned char parity[256];

unsigned long rec_count[MAXREC];
unsigned char rec_data[MAXREC];
unsigned long bad_parity[MAXREC];

int print_recinfo;
int print_errinfo;

int recnum;
int filenum;

int cur_filenum;
int cur_recnum;
int cur_recsize;
int cur_reccount;

char outfilename[1000];
FILE *outfile;

void set_filebase(char *fn);
void open_output(char *ext);
void process_record(int n);

void main(int argc, char **argv)
{
	FILE	*f;
	int i;
	unsigned long count;
	int c;
	int gap;
	int optind;

	/* Process arguments */

	print_recinfo = 0;
	print_errinfo = 0;

	optind = 1;
	while (argv[optind][0] == '-') {
		switch(argv[optind][1]) {
		case 'r':
			print_recinfo++;
			break;

		case 'e':
			print_errinfo++;
			break;
		}
		optind++;
	}

	/* Open files */

	if (optind != argc-1) {
		fprintf(stderr, "Usage: g7tdcode [-ree] <raw g7t tape file>\n");
		exit(1);
	}
	f = fopen(argv[optind], "rb");
	if (f == NULL) {
		perror(argv[optind]);
		exit(1);
	}
	set_filebase(argv[optind]);

	/* Initialize */

	filenum = 1;
	recnum = 0;

	for (i = 0; i < sizeof(parity); i++) {
		unsigned char b;
		int c;

		for (c = 0, b = 1; b; b <<= 1) {
			if (i & b) {
				c++;
			}
		}
		parity[i] = c & 1;
	}

	/* Process */

	gap = 0;	
	i = 0;
	for (;;) {
		c = fgetc(f);
		if (c == EOF) {
			break;
		}
		if (c < 0 || c >= 256) {
			fprintf(stderr, "Invalid character %d\n", c);
			break;
		}
		if (c != 0) {
			count = c;
			if (gap) {
				count += GAP_CORRECTION;
			}
			gap = 0;
		} else {
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count = c;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count |= c << 8;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count |= c << 16;
			if (count == 0) {
				fprintf(stderr, "Zero long count encountered\n");
				break;
			}
			if (count < 256) {
				fprintf(stderr, "Low long count encountered %d\n", count);
			}
			if (gap) {
				count += GAP_CORRECTION;
			}
			gap = 1;
		}
		count += INVISIBLE_COUNT;

		if (i > 0 && count > RECORD_GAP_COUNT/2) {
			process_record(i);
			i = 0;
		}

		c = fgetc(f);
		if (c == EOF) {
			fprintf(stderr, "Unexpected end of file\n");
			break;
		}
		if (c < 0 || c >= 256) {
			fprintf(stderr, "Invalid character %d\n", c);
			break;
		}

		if (i < MAXREC) {
			rec_count[i] = count;
			rec_data[i] = c;
		}
		i++;
	}
	process_record(i);

	fclose(f);
	fclose(outfile);

	/* Report */

}

void set_filebase(char *fn)
{
	char *p;

	strcpy(outfilename, fn);
	for (p = outfilename; *p && *p != '.'; p++);
	*p++ = '.';
	*p = '\0';

	outfile = NULL;
}

void open_output(char *ext)
{

	strcat(outfilename, ext);
	outfile = fopen(outfilename, "wb");
	if (outfile == NULL) {
		perror(outfilename);
		exit(1);
	}
}

void process_record(int n)
{
	int	avg;
	int i, j;
	int m;
	int t;
	int b;
	int np;
	int den;
	int even_count;
	int odd_count;
	unsigned char total_data;
	int fix;

	if (n <= 1) {
		if (print_errinfo) {
			printf("File %d record %d spurious character %02X deleted\n",
				filenum, recnum, rec_data[0]);
		}
		return;
	}
	if (n >= MAXREC) {
		if (print_errinfo) {
			printf("File %d record %d too long %d deleted\n",
				filenum, recnum, n);
			printf(" data ");
			for (i = 0; i < 8; i++) {
				printf(" %02X", rec_data[i]);
			}
			printf("\n");
		}
		return;
	}

	if (n == 2 && rec_data[0] == FILEMARK && rec_data[1] == FILEMARK) {
		filenum++;
		if (outfile != NULL) {
			fputc(FILEMARK+RECORDMARK, outfile);
			fputc(FILEMARK, outfile);
		}
		return;
	} else if (n == 2) {
		if (print_errinfo) {
			printf("File %d record %d short record %02X %02X deleted\n",
				filenum, recnum, rec_data[0], rec_data[1]);
		}
		return;
	}
	recnum++;

	avg = 0;
	for (i = 1; i < n-1; i++) {
		avg += rec_count[i];
	}
	avg /= n-2;

	m = 10000;
	for (i = 0; i < 3; i++) {
		t = avg - COUNTS_PER_INCH / density[i];
		if (t < 0) {
			t = -t;
		}
		if (t < m) {
			m = t;
			den = i;
		}
	}
	t = COUNTS_PER_INCH / density[den];
	for (i = 1; i < n; i++) {
		rec_count[i] = (rec_count[i] + (t >> 1)) / t;
	}

	if (rec_count[n-1] < LRC_GAP_CHAR) {
		if (print_errinfo) {
			printf("File %d record %d short LRC gap %d, adding new LRC\n",
				filenum, recnum, rec_count[n-1]);
		}
		rec_count[n] = LRC_GAP_CHAR;
		rec_data[n] = 0;
		n++;
	}
	rec_count[n-1] -= LRC_GAP_CHAR-1;

	fix = 0;
	for (i = 1; i < n; i++) {
		while (rec_count[i] > 1) {
			for (j = n; j > i+1; j--) {
				rec_count[j] = rec_count[j-1];
				rec_data[j] = rec_data[j-1];
			}
			rec_count[i+1] = 1;
			rec_data[i+1] = 0;
			rec_count[i]--;
			n++;
			fix++;
		}
	}
	if (fix) {
		if (print_errinfo) {
			printf("File %d record %d - %d missing characters\n",
				filenum, recnum, fix);
		}
	}

	even_count = 0;
	odd_count = 0;
	total_data = 0;
	for (i = 0; i < n-1; i++) {
		if (parity[rec_data[i]]) {
			odd_count++;
		} else {
			even_count++;
		}
		total_data |= rec_data[i];
	}

	t = 0;	
	for (i = 0; i < n; i++) {
		t ^= rec_data[i];
	}

	if (odd_count != 0 && even_count != 0 || t != 0) {
		if (print_errinfo) {
			printf("File %d record %d LRC %02X data error odd %d even %d\n",
				filenum, recnum, t, odd_count, even_count);
		}
		np = 0;
		if (odd_count > even_count) {
			for (i = 0; i < n-1; i++) {
				if (!parity[rec_data[i]]) {
					bad_parity[np++] = i;
					if (print_errinfo) {
						printf(" %6d. %02X\n", i, rec_data[i]);
					}
				}
			}
		} else {
			for (i = 0; i < n-1; i++) {
				if (parity[rec_data[i]]) {
					bad_parity[np++] = i;
					if (print_errinfo) {
						printf(" %6d. %02X\n", i, rec_data[i]);
					}
				}
			}
		}

		for (b = 1; b; b <<= 1) {
			if (b == t) {
				for (i = bad_parity[0]; i <= bad_parity[np-1]; i++) {
					if ((rec_data[i] & b) != 0) {
						break;
					}
				}
				if (i > bad_parity[np-1]) {
					if (print_errinfo) {
						printf(" Correcting bit %02X %d - %d\n",
							b, bad_parity[0], bad_parity[np-1]);
					}
					for (i = 0; i < np; i++) {
						rec_data[bad_parity[i]] |= b;
					}
				}
				break;
			}
		}
	}

	if (outfile == NULL) {
		if (odd_count > even_count) {
			if (print_recinfo) {
				printf("%d bpi odd parity\n", density[den]);
			}
			open_output("bin");
		} else {
			if (print_recinfo) {
				printf("%d bpi even parity\n", density[den]);
			}
			open_output("bcd");
		}
		for (i = 1; i < filenum; i++) {
			fputc(FILEMARK + RECORDMARK, outfile);
			fputc(FILEMARK, outfile);
		}
		cur_recnum = recnum;
		cur_reccount = 1;
		cur_recsize = n-1;
		cur_filenum = filenum;
	} else {
		if (cur_recsize == n-1) {
			cur_reccount++;
		} else {
			printf("%4d %6d  %6d x %6d\n",
				cur_filenum, cur_recnum, cur_recsize, cur_reccount);
			cur_filenum = filenum;
			cur_recnum = recnum;
			cur_recsize = n-1;
			cur_reccount = 1;
		}
	}

	rec_data[0] |= RECORDMARK;
	fwrite(rec_data, n-1, 1, outfile);

}