/*
 * Merge 7-track raw data files
 * to finished data file
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "../include/parity.h"
#include "../include/bitcount.h"

#define MAXREC	(6*32768+100)

int	num_good;
int	num_corr;
int	num_bad;

int recnum;

FILE *fo;

int nfile;
struct file {
	char *name;
	FILE *fi;
	int next;
	int len;
	int datalen;
	int parity;
	int good;
	int zero;
	int lrc;
	int bitcount;
	int ref;
	int match;
	unsigned char fix_bit;
	int first;
	int last;
	int rec_good;
	int rec_corr;
	int rec_used;
	int good_hist;
	int corr_hist;
	unsigned char *rec;
} *file;

struct lenset {
	unsigned long set;
	int len;
	int num;
} *lenset;

void mdat_init(int nfile, char **file_list, char *out_name)
{
	struct file *f;
	char s[100];
	int odd, even;
	int i;

	/*
	 * Allocate data structures
	 */

	file = (struct file *)malloc(3 * nfile * sizeof(struct file));
	if (file == NULL) {
		perror("malloc");
		exit(1);
	}
	lenset = (struct lenset *)malloc(nfile * sizeof(struct lenset));
	if (lenset == NULL) {
		perror("malloc");
		exit(1);
	}

	for (i = 0; i < 3 * nfile; i++) {
		f = &file[i];

		f->rec = (unsigned char *)malloc(MAXREC);
		if (f->rec == NULL) {
			perror("malloc");
			exit(1);
		}
		f->rec_good = 0;
		f->rec_corr = 0;
		f->rec_used = 0;
		f->good_hist = 0;
		f->corr_hist = 0;
	}

	/*
	 * Open and prime input files
	 */

	odd = 0;
	even = 0;
	for (i = 0; i < nfile; i++) {
		f = &file[i];

		f->name = file_list[i];
		f->fi = fopen(f->name, "rb");
		if (f->fi == NULL) {
			perror(f->name);
			exit(1);
		}
		f->next = fgetc(f->fi);
		if ((f->next & 0x80) == 0) {
			fprintf(stderr, "Bad format %s\n", f->name);
			exit(1);
		}
		if (parity[f->next & 0x7F]) {
			odd++;
		} else {
			even++;
		}
	}

	/*
	 * Open output file
	 */

	strcpy(s, out_name);
	if (odd >= even) {
		strcat(s, ".bin");
	} else {
		strcat(s, ".bcd");
	}
	fo = fopen(s, "wb");
	if (fo == NULL) {
		perror(s);
		exit(1);
	}
}

int mdat_read(struct file *f)
{
	int c;

	f->len = 0;
	if (f->next != EOF) {
		c = f->next;
		for (;;) {
			f->rec[f->len++] = c & 0x7F;
			c = fgetc(f->fi);
			if (c == EOF || (c & 0x80) != 0) {
				break;
			}
		}
		f->next = c;
		return 1;
	} else {
		return 0;
	}
}

void mdat_eval(struct file *f)
{
	int	odd, even;
	int j;

	/*
	 * See if there is an LRC
	 */

	if (f->len > 3 &&
		f->rec[f->len-1] != 0 &&
		f->rec[f->len-2] == 0 &&
		f->rec[f->len-3] == 0) {
		f->datalen = f->len - 3;
		while (f->datalen > 1 && f->rec[f->datalen-1] == 0) {
			f->datalen--;
		}
	} else {
		f->datalen = f->len;
	}

	/*
	 * Count parity and calculate LRC error
	 */

	f->lrc = 0;
	f->bitcount = 0;
	for (j = 0; j < f->len; j++) {
		f->lrc ^= f->rec[j];
		f->bitcount += bitcount[f->rec[j]];
	}

	odd = even = 0;
	for (j = 0; j < f->datalen; j++) {
		if (f->rec[j] != 0) {
			if (parity[f->rec[j]]) {
				odd++;
			} else {
				even++;
			}
		}
	}
	f->parity = odd >= even;

	/*
	 * Mark good characters
	 * and count zeroes
	 */

	f->good = 0;
	f->zero = 0;
	for (j = 0; j < f->datalen; j++) {
		if (f->rec[j] != 0) {
			if (parity[f->rec[j]] == f->parity) {
				f->good++;
			}
		} else {
			f->zero++;
		}
	}

	f->fix_bit = 0;
	f->first = -1;
	f->last = -1;
}

int mdat_correct(struct file *f)
{
	int first, last;
	int j;
	unsigned char b;

	/*
	 * Delimit the scope of bad parity
	 */

	first = -1;
	last = -1;
	for (j = 0; j < f->datalen; j++) {
		if (parity[f->rec[j]] != f->parity) {
			if (first == -1) {
				first = j;
			}
			last = j;
		}
	}
	f->first = first;
	f->last = last;

	/*
	 * See if its easy to fix - only one bit dropped throughout
	 */

	if (first >= 0) for (b = 1; b; b <<= 1) {
		if (b == f->lrc) {
			for (j = first; j <= last; j++) {
				if ((f->rec[j] & b) != 0) {
					break;
				}
			}
			if (j > last) {
				for (j = first; j <= last; j++) {
					if (parity[f->rec[j]] != f->parity) {
						f->rec[j] |= b;
						f->bitcount++;
					}
				}
				f->fix_bit = b;
				f->lrc = 0;
				f->bitcount--;
				f->good = f->datalen;
				return 1;
			}
		}
	}
	return 0;
}

int mdat_best(int n)
{
	struct file *f;
	int match, ok, lrcbits;
	int good, max_data, min_lrcbits, max_match, max_bits;
	int best, even_better;
	int	i, j, k;


	/*
	 * See which records match each other
	 */

	for (i = 0; i < n; i++) {
		f = &file[i];

		f->ref = -1;
		f->match = 1;

		for (k = 0; k < i; k++) {
			if (file[k].ref == -1) {
				j = 0;
				if (file[k].datalen == f->datalen) {
					for (; j < f->datalen; j++) {
						if (file[k].rec[j] != f->rec[j]) {
							break;
						}
					}
				}
				if (j == f->datalen) {
					f->ref = k;
					file[k].match++;
					f->match = 0;
					break;
				}
			}
		}
	}

	/*
	 * See which was the best
	 */

	good = max_data = max_match = max_bits = best = 0;
	min_lrcbits = 7;
	for (i = 0; i < n; i++) {
		f = &file[i];
		ok = f->good == f->datalen && f->lrc == 0;
		match = (f->ref == -1)? f->match : file[f->ref].match;
		lrcbits = bitcount[f->lrc];

		if (ok > good ||
			ok == good &&
			(f->bitcount > max_bits ||
			 f->bitcount == max_bits &&
			 (f->good > max_data ||
			  f->good == max_data &&
			  (lrcbits < min_lrcbits ||
			   lrcbits == min_lrcbits &&
			    match > max_match)))) {
			good = ok;
			max_data = f->good;
			min_lrcbits = lrcbits;
			max_bits = f->bitcount;
			max_match = match;
			best = i;
		}
	}
	if (max_match > 0) {
		for (k = best + 1; k < n; k++) {
			if (file[k].bitcount == max_bits && file[k].match == max_match) {
				printf("%6d. %s apparently good record different, tie\n",
					recnum, file[k].name);
			}
		}

		/*
		 * Find the shortest correction
		 * Count our successes
		 */

		even_better = best;
		for (k = best; k < n; k++) {
			if (k == best || file[k].ref == best) {
				f = &file[k];
				if (f->fix_bit == 0) {
					f->rec_good++;
					if (file[even_better].fix_bit != 0) {
						even_better = k;
					}
				} else {
					f->rec_corr++;
					if (file[even_better].fix_bit != 0 &&
						f->last - f->first <
						file[even_better].last - file[even_better].first) {
						even_better = k;
					}
				}
			}
		}
		best = even_better;

		if (file[best].fix_bit == 0) {
			file[max_match-1].good_hist++;
		} else {
			file[max_match-1].corr_hist++;
		}
		file[best].rec_used++;

		return best;
	}
	return -1;
}

void mdat_construct_lenset()
{
	int i, n;
	int minlen;
	struct lenset *ls;

	minlen = 0;
	for (n = 0; ; n++) {
		ls = &lenset[n];
		ls->set = 0;
		ls->len = 0;
		ls->num = 0;
		for (i = 0; i < nfile; i++) {
			if (ls->len == 0) {
				if (file[i].len > minlen) {
					ls->len = file[i].len;
					ls->set |= 1 << i;
					ls->num++;
				}
			} else {
				if (file[i].len == ls->len) {
					ls->set |= 1 << i;
					ls->num++;
				}
			}
		}
		if (ls->num == 0) {
			break;
		}
		minlen = ls->len;
	}
}

void mdat_construct_or(int fi, int len)
{
	struct file *f;
	struct file *file_or;
	int i, j;

	file_or = &file[fi];
	file_or->name = "OR";
	file_or->len = len;
	for (j = 0; j < len; j++) {
		file_or->rec[j] = 0;
	}
	for (i = 0; i < nfile; i++) {
		f = &file[i];

		if (f->len != len) {
			continue;
		}
		for (j = 0; j < len; j++) {
			file_or->rec[j] |= f->rec[j];
		}
	}
}

void mdat_construct_merge(int fi, int len)
{
	struct file *file_merge;
	struct file *f;
	unsigned char c;
	int good, bits;
	int ok, more;
	int i, j;

	file_merge = &file[fi];
	file_merge->name = "Merge";
	for (j = 0; j < len; j++) {
		c = 0;
		good = 0;
		bits = 0;
		for (i = 0; i < fi; i++) {
			f = &file[i];

			if (f->len != len) {
				continue;
			}

			ok = parity[f->rec[j]] == f->parity;
			more = bitcount[f->rec[j]] > bits;
			if (!good && (ok | more) ||
				ok && more) {
				c = f->rec[j];
				bits = bitcount[c];
				good = ok;
			}
		}
		file_merge->rec[j] = c;
	}
	file_merge->len = len;
}

void mdat_dumprec(int n)
{
	struct file *f;
	int i, j;
	int m;

	printf("\n   char ");
	for (i = 0; i < n; i++) {
		printf(" %12s", file[i].name);
	}
	printf("\n");

	m = 0;
	for (i = 0; i < n; i++) {
		if (file[i].len > m) {
			m = file[i].len;
		}
	}
	for (j = 0; j < m; j++) {
		printf("%6d. ", j);
		for (i = 0; i < n; i++) {
			f = &file[i];

			if (f->datalen > j) {
				printf("       %02X%c   ",
					f->rec[j],
					parity[f->rec[j]] != f->parity? '*' : ' ');
			} else {
				printf("             ");
			}
		}
		printf("\n");
	}

	printf("   lrc  ");
	for (i = 0; i < n; i++) {
		f = &file[i];

		if (f->datalen < f->len) {
			printf("       %02X    ",
				f->rec[f->len-1]);
		} else {
			printf("             ");
		}
	}
	printf("\n");

	printf("lrcerr  ");
	for (i = 0; i < n; i++) {
		f = &file[i];

		printf("       %02X    ",
				f->lrc);
	}
	printf("\n");

	printf("\n");
}

void mdat_write(struct file *f)
{
	int j;

	/*
	 * Write it out
	 */

	f->rec[0] |= 0x80;
	for (j = 0; j < f->datalen; j++) {
		fputc(f->rec[j], fo);
	}
}

void main(argc, argv)
	int	argc;
	char **argv;
{
	int optind;
	char *optarg;
	int dumpflag;
	int fin;
	struct file *f;
	int n;
	int m;
	int i;

	if (argc < 3) {
usage:
		fprintf(stderr, "Usage: g7tmdat [-d] <outname> (<infile>...)\n");
		exit(1);
	}

	dumpflag = 0;
	for (optind = 1; optind < argc && argv[optind][0] == '-'; optind++) {
		for (optarg = &argv[optind][1]; *optarg; ) {
			switch (*optarg++) {

			case 'd':
				dumpflag = 1;
				break;

			default:
				goto usage;
			}
		}
	}
	fin = optind + 1;

	nfile = argc - fin;
	mdat_init(nfile, &argv[fin], argv[optind]);

	/*
	 * Process data
	 */

	num_good = 0;
	num_corr = 0;
	num_bad = 0;

	m = 0;
	recnum = 0;
	for (;;) {
		int go;

		recnum++;

		/*
		 * Read next records
		 */

		go = 0;
		for (i = 0; i < nfile; i++) {
			go |= mdat_read(&file[i]);

		}
		if (go == 0) {
			break;
		}

		/*
		 * Now make record(s) that is the OR of the others
		 */

		mdat_construct_lenset();
		n = nfile;
		for (i = 0; lenset[i].num != 0; i++) {
			if (lenset[i].num > 1) {
				mdat_construct_or(n, lenset[i].len);
				n++;
			}
		}

		/*
		 * Evaluate input and OR'ed records
		 */

		for (i = 0; i < n; i++) {
			mdat_eval(&file[i]);
		}

		/*
		 * Make record(s) of good characters merged from the others
		 */

		for (i = 0; lenset[i].num != 0; i++) {
			if (lenset[i].num > 1) {
				mdat_construct_merge(n, lenset[i].len);
				mdat_eval(&file[n]);
				n++;
			}
		}

		/*
		 * Attempt correction of those in error
		 */

		for (i = 0; i < n; i++) {
			if (file[i].lrc != 0) {
				mdat_correct(&file[i]);
			}
		}

		/*
		 * See if there are any good ones
		 */

		i = mdat_best(n);
		f = &file[i];
		if (f->good == f->datalen && f->lrc == 0) {
			if (f->fix_bit == 0) {
				num_good++;
			} else {
				printf("%6d. %12s corrected bit %02X %d - %d\n",
						recnum, f->name, f->fix_bit, f->first, f->last);
				num_corr++;
			}

		} else {

			num_bad++;

			printf("%6d. %12s %5d bad  lrc %02X",
					   recnum, f->name, f->datalen - f->good, f->lrc);
			if (f->first != -1 && f->last - f->first < 6) {
				printf(" ");
				for (i = (f->first > 0)? f->first-1 : 0;
					 i < ((f->last+1 < f->datalen)? f->last+2 : f->datalen);
					 i++) {
					printf(" %02X", f->rec[i]);
				}
			}
			printf("\n");
			if (dumpflag) {
				mdat_dumprec(n);
			}
		}
		if (n > m) {
			m = n;
		}

		mdat_write(f);
	}
	fclose(fo);

	printf("\n    File        Good   Corr   Used     Hist   Good   Corr\n");
	for (i = 0; i < m; i++) {
		f = &file[i];

		printf("%12s %6d %6d %6d  %6d. %6d %6d\n",
			f->name, f->rec_good, f->rec_corr, f->rec_used,
			i+1, f->good_hist, f->corr_hist);
	}

	printf("\nRecords %d  good %d  corrected %d  bad %d\n",
		recnum-1, num_good, num_corr, num_bad);
	exit(0);
}