#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>

#define MAXREC			(32768*6 + 1000)

#define FILEMARK		0x0F
#define RECORDMARK			0x80

#define COUNTS_PER_INCH_FWD	30000
#define COUNTS_PER_INCH_REV 27500
#define INVISIBLE_COUNT	5
#define GAP_CORRECTION	1

#define LRC_GAP_CHAR	4
#define RECORD_GAP_COUNT 5000

int counts_per_inch;
int density[] = { 200, 556, 800 };

unsigned long rec_count[MAXREC];
unsigned char rec_data[MAXREC];

int print_recinfo;
int print_errinfo;

int reverse;

int recnum;

int cur_recnum;
int cur_recsize;
int cur_reccount;

int total_chars;
int total_recs;

char outfilename[1000];
FILE *outfile;

void set_filebase(char *fn);
void open_output(char *ext);
void process_record(int n);

void main(int argc, char **argv)
{
	FILE	*f;
	int i;
	unsigned long count;
	int c;
	int gap;
	int optind;

	/* Process arguments */

	print_recinfo = 0;
	print_errinfo = 0;

	optind = 1;
	counts_per_inch = -1;
	while (argv[optind][0] == '-') {
		switch(argv[optind][1]) {
		case 'r':
			print_recinfo++;
			break;

		case 'e':
			print_errinfo++;
			break;

		case 'c':
			counts_per_inch = atoi(argv[optind+1]);
			optind++;
			break;
		}
		optind++;
	}

	/* Open files */

	if (optind != argc-1) {
		fprintf(stderr, "Usage: g7tdtim [-rec] <raw g7t tape file>\n");
		exit(1);
	}
	f = fopen(argv[optind], "rb");
	if (f == NULL) {
		perror(argv[optind]);
		exit(1);
	}
	set_filebase(argv[optind]);
	if (reverse) {
		open_output("rrd");
	} else {
		open_output("frd");
	}

	/* Initialize */

	recnum = 0;
	cur_recsize = -1;
	cur_reccount = 0;
	total_chars = 0;
	total_recs = 0;
	if (counts_per_inch == -1) {
		counts_per_inch = reverse? COUNTS_PER_INCH_REV : COUNTS_PER_INCH_FWD;
	}

	/* Process */

	gap = 0;	
	i = 0;
	for (;;) {
		c = fgetc(f);
		if (c == EOF) {
			break;
		}
		if (c < 0 || c >= 256) {
			fprintf(stderr, "Invalid character %d\n", c);
			break;
		}
		if (c != 0) {
			count = c;
			if (gap) {
				count += GAP_CORRECTION;
			}
			gap = 0;
		} else {
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count = c;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count |= c << 8;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			if (c < 0 || c >= 256) {
				fprintf(stderr, "Invalid character %d\n", c);
				break;
			}
			count |= c << 16;
			if (count == 0) {
				fprintf(stderr, "Zero long count encountered\n");
				break;
			}
			if (count < 256) {
				fprintf(stderr, "Low long count encountered %d\n", count);
			}
			if (gap) {
				count += GAP_CORRECTION;
			}
			gap = 1;
		}
		count += INVISIBLE_COUNT;

		if (i > 0 && count > RECORD_GAP_COUNT) {
			process_record(i);
			i = 0;
		}

		c = fgetc(f);
		if (c == EOF) {
			fprintf(stderr, "Unexpected end of file\n");
			break;
		}
		if (c < 0 || c >= 256) {
			fprintf(stderr, "Invalid character %d\n", c);
			break;
		}

		if (i < MAXREC) {
			rec_count[i] = count;
			rec_data[i] = c;
		}
		i++;
	}
	if (i > 0) {
		process_record(i);
	}

	if (cur_reccount > 0) {
		if (print_recinfo) {
			printf("%6d.  %6d x %6d\n",
				cur_recnum, cur_recsize, cur_reccount);
		}
	}

	if (print_recinfo) {
		printf("Total records %d  characters %d\n", total_recs, total_chars);
	}
	fclose(f);
	fclose(outfile);

}

void set_filebase(char *fn)
{
	char *p;

	strcpy(outfilename, fn);
	for (p = outfilename; *p && *p != '.'; p++);
	if (*p == '.' &&
		(p[1] == 'r' || p[1] == 'R') &&
		(p[2] == 'e' || p[2] == 'E') &&
		(p[3] == 'v' || p[3] == 'V')) {
		reverse = 1;
	}
	*p++ = '.';
	*p = '\0';

	outfile = NULL;
}

void open_output(char *ext)
{

	strcat(outfilename, ext);
	outfile = fopen(outfilename, "wb");
	if (outfile == NULL) {
		perror(outfilename);
		exit(1);
	}
}

void process_record(int n)
{
	int	avg;
	int i, j;
	int m;
	int t;
	int den;
	static int old_den = -1;

	recnum++;

	if (n >= MAXREC) {
		if (print_errinfo) {
			printf("Record %d too long %d deleted\n",
				recnum, n);
			printf(" data ");
			for (i = 0; i < 8; i++) {
				printf(" %02X", rec_data[i]);
			}
			printf("\n");
		}
		return;
	}

	den = old_den;
	avg = 0;
	for (i = 1; i < n-1; i++) {
		avg += rec_count[i];
	}
	if (n > 2) {
		avg /= n-2;

		m = 10000;
		for (i = 0; i < 3; i++) {
			t = avg - counts_per_inch / density[i];
			if (t < 0) {
				t = -t;
			}
			if (t < m) {
				m = t;
				den = i;
			}
		}
		t = counts_per_inch / density[den];
	} else if (den == -1) {
		t = rec_count[1] / LRC_GAP_CHAR;
		if (t == 0) {
			t = 1;
		}
	} else {
		t = counts_per_inch / density[den];
	}
	for (i = 1; i < n; i++) {
		rec_count[i] = (rec_count[i] + (t >> 1)) / t;
	}

	for (i = 1; i < n; i++) {
		while (rec_count[i] > 1) {
			for (j = n; j > i; j--) {
				rec_count[j] = rec_count[j-1];
				rec_data[j] = rec_data[j-1];
			}
			rec_count[i] = 1;
			rec_data[i] = 0;
			i++;
			rec_count[i]--;
			n++;
		}
	}

	if (den != old_den) {
		if (print_recinfo || print_errinfo) {
			printf("Density %d %s\n", density[den],
				(old_den == -1 && reverse)? "reverse" : "");
		}
		old_den = den;
	}

	if (cur_recsize == -1) {
		cur_recnum = recnum;
		cur_reccount = 1;
		cur_recsize = n;
	} else {
		if (cur_recsize == n) {
			cur_reccount++;
		} else {
			if (print_recinfo) {
				printf("%6d.  %6d x %6d\n",
					cur_recnum, cur_recsize, cur_reccount);
			}
			cur_recnum = recnum;
			cur_recsize = n;
			cur_reccount = 1;
		}
	}
	total_chars += n;
	total_recs++;

	rec_data[reverse? n-1 : 0] |= RECORDMARK;
	fwrite(rec_data, n, 1, outfile);
}