#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define SMALL_LC	40000
#define BIG_LC		1500
#define BIG_LC_BUCK	1000

unsigned long sc_hist[256];
unsigned long scag_hist[256];
unsigned long lc_hist[SMALL_LC];
unsigned long lcag_hist[SMALL_LC];
unsigned long lc_buck[BIG_LC];

unsigned long even_count;
unsigned long odd_count;

unsigned char total_data;

unsigned char parity[256];


void stats_banner();
void stats_scan(char *s, unsigned long *hist, int n, int z, int w);
void dostats(char *s, unsigned long *hist, int first, int last, int w);

void main(int argc, char **argv)
{
	FILE	*f;
	int i;
	unsigned long count;
	int c;
	int gap;

	/* Open files */

	if (argc != 2) {
		fprintf(stderr, "Usage: g7tstat <raw g7t tape file>\n");
		exit(1);
	}
	f = fopen(argv[1], "rb");
	if (f == NULL) {
		perror(argv[1]);
		exit(1);
	}

	/* Initialize */

	for (i = 0; i < 256; i++) {
		sc_hist[i] = 0;
		scag_hist[i] = 0;
	}
	for (i = 0; i < sizeof(lc_hist)/sizeof(lc_hist[0]); i++) {
		lc_hist[i] = 0;
		lcag_hist[i] = 0;
	}
	for (i = 0; i < sizeof(lc_buck)/sizeof(lc_buck[0]); i++) {
		lc_buck[i] = 0;
	}
	even_count = 0;
	odd_count = 0;
	total_data = 0;

	for (i = 0; i < sizeof(parity); i++) {
		unsigned char b;
		int c;

		for (c = 0, b = 1; b; b <<= 1) {
			if (i & b) {
				c++;
			}
		}
		parity[i] = c & 1;
	}

	/* Process */

	gap = 0;	
	for (;;) {
		c = fgetc(f);
		if (c == EOF) {
			break;
		}
		if (c != 0) {
			count = c;
			if (gap) {
				scag_hist[count]++;
			} else {
				sc_hist[count]++;
			}
			gap = 0;
		} else {
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			count = c;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			count |= c << 8;
			c = fgetc(f);
			if (c == EOF) {
				fprintf(stderr, "Unexpected end of file\n");
				break;
			}
			count |= c << 16;
			if (count == 0) {
				fprintf(stderr, "Zero long count encountered\n");
				break;
			}
			if (count < 256) {
				fprintf(stderr, "Low long count encountered %d\n", count);
			}

			if (count < SMALL_LC) {
				if (gap) {
					lcag_hist[count]++;
				} else {
					lc_hist[count]++;
				}
			} else {
				if (count >= BIG_LC*BIG_LC_BUCK) {
					fprintf(stderr, "Overrange count encountered: %08X\n", count);
					break;
				}
				lc_buck[count / BIG_LC_BUCK]++;
			}
			gap = 1;
		}
		c = fgetc(f);
		if (c == EOF) {
			fprintf(stderr, "Unexpected end of file\n");
			break;
		}
		if (parity[c]) {
			odd_count++;
		} else {
			even_count++;
		}
		total_data |= c;
	}
	fclose(f);

	/* Report */

	printf("Even %d  Odd %d  Bits %02X\n", even_count, odd_count, total_data);

	if (sc_hist[255] > 0 && lc_hist[256] > 0) {
		for (i = 255; sc_hist[i]; --i) {
			lc_hist[i] = sc_hist[i];
			sc_hist[i] = 0;
		}
	}

	if (scag_hist[255] > 0 && lcag_hist[256] > 0) {
		for (i = 255; scag_hist[i]; --i) {
			lcag_hist[i] = scag_hist[i];
			scag_hist[i] = 0;
		}
	}

	stats_banner();
	stats_scan("SC  ", sc_hist, 256, 3, 1);
	stats_scan("MC  ", lc_hist, SMALL_LC, 1000, 1);
	stats_scan("LC  ", lc_buck, BIG_LC, 3, BIG_LC_BUCK);
	printf("\n");
	stats_scan("SCAG", scag_hist, 256, 3, 1);
	stats_scan("MCAG", lcag_hist, SMALL_LC, 1000, 1);
}

void stats_banner()
{
	printf("\nStat    Count     Low   Median   High      Avg        SD\n\n");
}

void stats_scan(char *s, unsigned long *hist, int n, int z, int w)
{
	int i, j, k;

	for (i = 0; i < n; ) {
		if (hist[i] == 0) {
			i++;
			continue;
		}
		for (j = i; j < n; j++) {
			for (k = j; k < j+z; k++) {
				if (hist[k] != 0) {
					break;
				}
			}
			if (k == j+z) {
				break;
			}
		}
		dostats(s, hist, i, j, w);
		i = j;
	}
}

void dostats(char *s, unsigned long *hist, int first, int last, int w)
{
	int	i;
	int sum;
	int count;
	int median;
	int mc;
	double avg;
	double dev;
	double t;

	sum = 0;
	count = 0;
	for (i = first; i < last; i++) {
		sum += w*i*hist[i];
		count += hist[i];
	}
	avg = (double)sum / (double)count;

	dev = 0.0;
	mc = 0;
	median = first;
	for (i = first; i < last; i++) {
		t = (double)w*i - avg;
		dev += hist[i] * t*t;
		mc += hist[i];
		if (mc <= count/2) {
			median = i;
		}
	}
	dev = sqrt(dev / count);

	printf("%4s %8d %7d %7d %7d %9.1f %9.1f\n",
			s, count, w*first, w*median, w*(last-1), avg, dev);
}